proc(b,bv,stat)=glmest(code,x,y,opt) 
; ----------------------------------------------------------------------------
; Library        glm
; ----------------------------------------------------------------------------
;  See_also      glmopt glminit glmcore glmstat glmest doglm
; ----------------------------------------------------------------------------
;   Macro        glmest
; ----------------------------------------------------------------------------
;   Description  glmest fits a  generalized linear model E[y|x] = G(x*b).
;                This macro offers a convenient interface for GLM estimation.
;                A check of the data is performed. 
; ----------------------------------------------------------------------------
;   Keywords     GLM, Generalized Linear Model
; ----------------------------------------------------------------------------
;   Reference    McCullagh/Nelder, Generalized Linear Models, 1989.
; ----------------------------------------------------------------------------
;   Link   ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        myfit = glmest(code,x,y{,opt})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset. Can be used for
;                       constrained estimation. If not given, set to 0.
;     Parameter   opt.shf  
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the iteration is going on.
;     Parameter   opt.norepl
;     Definition        integer, if exists and =1, the data are assumed to 
;                       have no replications in x. Otherwise, the data
;                       are searched for replications to fasten the
;                       algorithm.
;     Parameter   opt.miter
;     Definition        integer, maximal number of iterations. The default 
;                       is 10.
;     Parameter   opt.cnv
;     Definition        scalar, convergence criterion. The default is 0.0001.
;     Parameter   opt.fscor
;     Definition        integer, if exists and =1, a Fisher scoring is
;                       performed (instead of the default Newton-Raphson
;                       procedure). This parameter is ignored for
;                       canonical links.
;     Parameter   opt.pow
;     Definition        scalar, power for power link. If not given,
;                       set to 0.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   myfit.b
;     Definition        p x 1 vector, estimated coefficients.
;     Parameter   myfit.bv
;     Definition        p x p matrix, estimated covariance matrix 
;                       for coefficients.
;     Parameter   myfit.stat
;     Definition        list with components as computed by glmstat:
;                       serror     (standard errors of coefficients), 
;                       tvalue     (t-values for coefficients), 
;                       pvalue     (p-values for coefficients), 
;                       df         (degrees of freedom),
;                       deviance   (deviance),
;                       pearson    (generalized pearson's chi^2),
;                       loglik     (log-likelihood),
;                       dispersion (estimated dispersion =pearson/df),
;                       r2         ((pseudo) coefficient of determination),
;                       adr2       (adjusted (pseudo) coefficient of determination),
;                       aic        (Akaike's AIC criterion),
;                       bic        (Schwarz' BIC criterion), and
;                       it         (number of iterations needed),
;                       ret        (return code, 
;                                     0 if everything went o.k., 
;                                     1 if maximal number of iterations reached, 
;                                    -1 if missing values have been encountered),
;                       nr         (number of replications found in x).
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             y=( 1./(1+exp(-x*b)).>uniform(n) )
;             ;==========================
;             ;  GLM fit 
;             ;==========================
;             opt=glmopt("shf",1)
;             lf=glmest("bilo",x,y,opt)
;             b~lf.b
;             lf.stat.deviance
; ----------------------------------------------------------------------------
;   Result    A logit fit for E[y|x] is computed. The option "shf" for
;             information during the iteration is set. lf.b contains the 
;             coefficients for the linear part. The example gives the 
;             true b together with the GLM estimate lf.b. The deviance 
;             of the fitted model is shown as well.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
; classify our algo
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  direct    = sum(code==(glmmodels.direct)) >0
  power     = sum(code==(glmmodels.power)) >0
  twoparfam = sum(code==(glmmodels.twoparfam)) >0
;
; Do we have options?
;
  norepl=0
  if (exist(opt)==0)
    dummy=0
    opt=list(dummy)  ; dummy optional parameter, doesn't matter
  else
    if (comp(opt,"norepl")>0)
      norepl=(opt.norepl==1)*(dim(dim(opt.norepl))==1)
    endif
  endif
;
; now we can run glminit
;
  {xr,yr,wxr,offr,ctrl}=glminit(code,x,y,opt)
;
; everything is now prepared for the estimation. run it!
;
  {b,bv,it,ret}=glmcore(code,xr,yr,wxr,offr,ctrl)
  nr=rows(xr)
;
; produce the statistics of what we have done. always with original x,y!
; note: bv is still not rescaled by dispersion!!
;
  stat=glmstat(code,x,y,b,bv,opt)
;
; now adjust bv by dispersion if two-parameter family
;
  if (twoparfam)
    bv=stat.dispersion.*bv
  endif
;
; append auxiliary stuff to stat
;
  append(stat,it)
  append(stat,ret)
  if (norepl!=1)    
    append(stat,nr)
  endif
endp 




