proc(mu)=glmlink(code,eta,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmcore glmstat glmout glminvlink
; ----------------------------------------------------------------------------
;   Macro       glmlink
; ----------------------------------------------------------------------------
;   Description  glmlink computes the link function.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        mu = glmlink(code,eta{,opt})
;   Input
;     Parameter   code 
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   eta  
;     Definition        n x d  matrix, the index values.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;     Parameter   opt.pow
;     Definition        optional, power for power link. If not given, set 
;                       to 0 (logarithm).
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   mu
;     Definition        n x d  matrix, the link  function applied to eta.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             eta=2
;             mu=glmlink("bilo",eta)
;             mu
; ----------------------------------------------------------------------------
;   Result    The logistic link for the logit model is computed at eta=2:
;             Contents of mu
;             [1,]   0.8808 
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
  binomial  = sum(code==(glmmodels.binomial)) >0
  power     = sum(code==(glmmodels.power)) >0
  identity  = sum(code==("noid")) >0
;
  switch
    case (binomial)
      switch
        case (code=="bilo")  
          e=exp(eta)
          mu=e./(1+e)   
          mu = mu+ 1e-16.*(mu==0) - 1e-16.*(mu==1)
          break
        case (code=="bipro")  
          mu = cdfn(eta) 
          mu = mu+ 1e-16.*(mu==0) - 1e-16.*(mu==1)
          break
        default ; code=="bicll"  
          e=exp(eta)
          mu   = 1-exp(-e)
          mu = mu+ 1e-16.*(mu==0) - 1e-16.*(mu==1)
      endsw
      break
    case (identity)
      mu=eta
      break
    case (power) 
      p=0
      if (exist(opt)==9)
        if (comp(opt,"pow")>0)
          p=opt.pow
        endif
      endif
      if (p==0)
        mu=exp(eta)
      else
        mu=eta^(1/p)
      endif
      break
    case (code=="polog")
      mu=exp(eta)
      break
    case (code=="gacl")
      mu=1./eta
      break
    case (code=="igcl")
      mu=1./sqrt(eta)
      break
    case (code=="nbcl")
      k=1
      if (exist(opt)==9)
        if (comp(opt,"nbk")>0)
          k=opt.nbk
        endif
      endif
      e=exp(eta)
      mu=e./(k.*(1-e))
      break
    default 
      error(0==0, "glm: '"+code+"' is not a valid model!")
      break
  endsw
endp




