proc(ll)=glmll(code,mu,y,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmcore glmlld glmstat
; ----------------------------------------------------------------------------
;   Macro       glmll
; ----------------------------------------------------------------------------
;   Description  glmll computes the individual log-likelihood.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        ll = glmll(code,mu,y{,opt})
;   Input
;     Parameter   code 
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary OLS).
;     Parameter   mu  
;     Definition        n x 1, n x 2 or n x 3 matrix, either the response 
;                       function, or sums of response function in the first 
;                       column.
;                       In second and third column expressions needed for 
;                       computation of deviance (typically sums of
;                       mu^2 or log(mu)).
;     Parameter   y  
;     Definition        n x 1, n x 2 or n x 3 matrix, either the response 
;                       values, or sums of response values in the first 
;                       column.
;                       In second and third column expressions needed for 
;                       computation of deviance (typically sums of
;                       y^2 or log(y)).
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;     Parameter   opt.phi
;     Definition        nuisance parameter, usually the dispersion 
;                       parameter. If not given, only those parts of the 
;                       log-likelihood are computed, which depend on the 
;                       parameter of interest.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   ll  
;     Definition        n x 1  matrix, log-likelihood.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             y=1
;             mu=0.75
;             ll=glmll("bilo",mu,y)
;             ll
; ----------------------------------------------------------------------------
;   Result    The log-likelihood for logit at y=1 and mu=0.75: 
;             Contents of ll
;             [1,] -0.28768
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  phi=NaN
  nbk=1
  if (exist(opt)>0)
    if (comp(opt,"phi")>0)
      phi=opt.phi
    endif
    if (comp(opt,"nbk")>0)
      nbk=opt.nbk
    endif
  endif

  switch
;
    case (binomial)  
;     ========================================================
;     there is only one case to handle :-)
;     ========================================================
      bad=(cols(y)>1) || (cols(mu)>1)
      error(bad,"y or mu has wrong dimension for glmll default case!")
      ll = xlny(y,mu)+xlny(1-y,1-mu)
      break
;
    case (poisson)  
;     ========================================================
;     put sum's of log(y) in second column of y if we have wx
;     ========================================================
      switch
        case ((cols(y)==2)*(cols(mu)==2))  ; handle glmll(code,y,y) with wx
          ll = y[,1].*mu[,2]-mu[,1]
          break
        case ((cols(y)==2)*(cols(mu)==1))  ; handle glmll(code,mu,y) with wx  
          ll = xlny(y[,1],mu[,1])-mu[,1]
          break
        default
          bad=(cols(y)>1) || (cols(mu)>1)
          error(bad,"y or mu has wrong dimension for glmll default case!")
          ll = xlny(y,mu)-mu - lgamma(y+1)
      endsw
      break
;
    case (gamma)  
;     ========================================================
;     put sum's of log(y) in second column of y if we have wx
;     ========================================================
      switch
        case ((cols(y)==2)*(cols(mu)==2))  ; handle glmll(code,y,y) with wx
          ll = -y[,1]./mu[,1] -mu[,2]
          break
        case ((cols(y)==2)*(cols(mu)==1))  ; handle glmll(code,mu,y) with wx  
          ll = -y[,1]./mu[,1] -log(mu[,1])
          break
        default
          bad=(cols(y)>1) || (cols(mu)>1)
          error(bad,"y or mu has wrong dimension for glmll default case!")
          ll = -y./mu - log(mu)
          if (isNumber(phi))
            ll = ll.*phi -lgamma(phi) +phi.*log(phi.*y) -log(y)
          endif
      endsw
      break
;
    case (igaussian)
;     ========================================================
;     put sum's of 1/y in second column of y if we have wx
;     ========================================================
      switch
        case ((cols(y)==2)*(cols(mu)==2))  ; handle glmll(code,y,y) with wx
          ll =  0.5./ y[,2]
          break
        case ((cols(y)==2)*(cols(mu)==1))  ; handle glmll(code,mu,y) with wx  
          ll = -0.5.* y[,1]./(mu[,1].*mu[,1]) -1./mu[,1]
          break
        default
          bad=(cols(y)>1) || (cols(mu)>1)
          error(bad,"y or mu has wrong dimension for glmll default case!")
          ll = -0.5.* y./(mu.*mu) - 1./mu
          if (isNumber(phi))
            ll = ll./phi -0.5*log(2*pi) -1.5*log(y) -0.5*log(phi) 
            ll = ll -1./(2*phi.*y)
          endif
     endsw 
     break
;
     case (nbinomial)
;     ================================================================
;     put sum's of log(y) in second column of y if we have wx
;     put sum's of log(y+p), p=k, in third column of y if we have wx
;     ================================================================
      nu=1/nbk
      switch
        case ((cols(y)==3)*(cols(mu)==3))  ; handle glmll(code,y,y) with wx
          ll = y[,1].*mu[,2] - y[,1].*mu[,3] -nu.*mu[,3]
          break
        case ((cols(y)==3)*(cols(mu)==1))  ; handle glmll(code,mu,y) with wx  
          ll = y[,1].*log(mu[,1]) - y[,1].*log(mu[,1]+nu) -nu.*log(mu[,1]+nu)
          break
        default
          bad=(cols(y)>1) || (cols(mu)>1)
          error(bad,"y or mu has wrong dimension for glmll default case!")
          ll = y.*log(mu) - y.*log(mu+nu) -nu.*log(mu+nu)
          if (nbk!=1)
            ll = ll+lgamma(y[,1]+nu)-lgamma(y[,1]+1)-lgamma(nu)+nu.*log(nu)
          endif
      endsw
      break
;
   default ; (gaussian)
;     ========================================================
;     put sum's of y^2 in second column if we have wx
;     ========================================================
      switch
        case ((cols(y)==2)*(cols(mu)==2))  ; handle glmll(code,y,y) with wx
          ll =-0.5*(y[,2]+mu[,2]-2.*y[,1].*mu[,1])
          break
        case ((cols(y)==2)*(cols(mu)==1))  ; handle glmll(code,mu,y) with wx
          ll =-0.5*(y[,2]+mu[,1]^2-2.*y[,1].*mu[,1])
          break
        default
          bad=(cols(y)>1) || (cols(mu)>1)
          error(bad,"y or mu has wrong dimension for glmll default case!")
          ll =-0.5*(y-mu)^2        ; handle everything without wx
          if (isNumber(phi))
            ll = ll./phi -0.5*log(2*pi) -0.5*log(phi)
          endif
          break
      endsw
;
  endsw
endp



