proc(best,bestcrit,bestord,bestordcrit,bestfit)=glmselect(code,x,y,opt) 
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmforward glmbackward glmest glmcore glmstat
; ----------------------------------------------------------------------------
;   Macro       glmselect
; ----------------------------------------------------------------------------
;   Description  glmselect performs a model selection by searching
;                the best of all subset models w.r.t. the AIC or BIC 
;                criterion.
;                Optionally a number of columns can be given, which are 
;                always included in the submodels.
; ----------------------------------------------------------------------------
;   Notes        The number of estimated models is (2^np)-1, when np
;                denotes the number of variables to select.
; ----------------------------------------------------------------------------
;   Link   ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        sel = glmselect(code,x,y,opt)
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;     Parameter   opt
;     Definition        optional, a list with optional input. 'glmselect' 
;                       allows the same optional parameters as 'glmest'. 
;                       Additionally, the following are supported.
;     Parameter   opt.fix
;     Definition        r x 1, r < p,  numbers of columns which should be 
;                       always included in the model.
;     Parameter   opt.shm
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the selection is going on.
;     Parameter   opt.crit
;     Definition        string, either "aic" or "bic", the selection criterion
;                       to use. If not given "aic" is used.
;   Output
;     Parameter   best  
;     Definition        p x 5, five best models.
;     Parameter   bestcrit
;     Definition        list, containing criteria for five best models:
;     Parameter   bestcrit.aic
;     Definition        1 x 5, AIC's for five best models.
;     Parameter   bestcrit.bic
;     Definition        1 x 5, BIC's for five best models.
;     Parameter   bestord  
;     Definition        p x p or p x (p-r) best models for every order.
;     Parameter   bestordcrit  
;     Definition        list, containing criteria for five best models:
;     Parameter   bestordcrit.aic
;     Definition        1 x p or 1 x (p-r), AIC's for five best models 
;                       for every order.
;     Parameter   bestordcrit.bic  
;     Definition        1 x p or 1 x (p-r), BIC's for five best models 
;                       for every order.
;     Parameter   bestfit  
;     Definition        GLM fit for best model, contains b, bv  and stat
;                       as computed by 'glmest'.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             ;========================== 
;             ;  simulate data  
;             ;========================== 
;             n=200 
;             b=1|2|0|0 
;             p=rows(b) 
;             x=normal(n,p)
;             y=x*b+normal(n)
;             ;========================== 
;             ;  GLM model selection
;             ;========================== 
;             shm=1
;             x=matrix(n)~x
;             opt=glmopt("shm",1)        ; show selection going
;             opt=glmopt("fix",1:2,opt)  ; always variables 1,2
;             modsel=glmselect("noid",x,y,opt)
;             modsel.best
;             modsel.bestcrit
; ----------------------------------------------------------------------------
;   Result    The best out of 7 possible submodels is determined. The 5 best
;             models and their AIC's and BIC's are shown.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
; classify our algo
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  direct    = sum(code==(glmmodels.direct)) >0
  power     = sum(code==(glmmodels.power)) >0
  twoparfam = sum(code==(glmmodels.twoparfam)) >0
;
; Do we have options?
;
  shm=0
  crit="aic"
;
  if (exist(opt)==0)
    dummy=0
    opt=list(dummy)  ; dummy optional parameter, doesn't matter
  else
    if (comp(opt,"shm")>0)
      notgood=(exist(opt.shm)!=1)
      warning (notgood>0, "opt.shm not a number, used default instead")
      if (notgood==0)
        shm=(opt.shm==1)*(dim(dim(opt.shm))==1)
      endif
    endif
    if (comp(opt,"fix")>0)
      notgood=(exist(opt.fix)!=1)
      notgood=notgood || (dim(dim(opt.fix))>2) 
      notgood=notgood || (rows(opt.fix)>cols(x)) || (cols(opt.fix)>1)
      notgood=notgood || (min(opt.fix)<1) || (max(opt.fix)>cols(x))
      error (notgood>0, "opt.fix is not appropriate")
      if (notgood==0)
        fix=opt.fix
      endif
    endif
    if (comp(opt,"crit")>0)
      notgood=(exist(opt.crit)!=2)
      notgood=notgood || (dim(dim(opt.crit))>2) 
      notgood=notgood || (sum(opt.crit==("aic"|"bic"))==0)
      error (notgood>0, "opt.crit not appropriate")
      if (notgood==0)
        crit=opt.crit
      endif
    endif
  endif
;
; now we can run glminit
;
  {xr,yr,wxr,offr,ctrl}=glminit(code,x,y,opt)
  if (comp(opt,"wx")>0)
    wx=opt.wx
  else
    wx=1
  endif
  if (comp(opt,"weights")>0)
    weights=opt.weights
  else
    weights="prior"
  endif
  if (comp(opt,"off")>0)
    off=opt.off
  else
    off=0
  endif
  pow=ctrl[5]
  nbk=ctrl[6]
;
; everything is now prepared for the procedure...
;
  n =rows(x)
  nr=rows(xr)
  p =cols(x)
;
  if (exist(fix))
    tmp=discrete(fix)
    fix=tmp.xr
;
    if (min(fix)==0)
      fix=paf(fix,fix!=0)
    endif
  endif
;
;
  havework=0
  if (exist(fix))  
    if (rows(fix)<p)
      notfix=(1:p)
      notfix=paf(notfix,min(abs(notfix-fix'),2)>0)
      np=rows(notfix)
      models=grid(0.*matrix(np),matrix(np),2.*matrix(np))
      models=models[2:rows(models)]
      bestmodord=0.*matrix(np,p+2)
      fullmodel=fix|notfix
      havework=1
    endif
  else
    notfix=(1:p)
    np=rows(notfix)
    models=grid(0.*matrix(np),matrix(np),2.*matrix(np))
    models=models[2:rows(models)]
    bestmodord=0.*matrix(np,p+2)
    fullmodel=notfix
    havework=1
  endif
;
  if (havework)
    j=0
    jmax=(2^np-1)
    while (j< jmax)
      j=j+1
      if (shm)
        "'glmselect': "+string("%1.0f",j)+" of "+string("%1.0f",jmax)
      endif
      imodel=models[j]
      model=(imodel'.*notfix)
      model=paf(model,model!=0)
      if (exist(fix))
        model=fix|model
      endif
      {b,bv,it,ret}=glmcore(code,xr[,model],yr,wxr,offr,ctrl)
      stat=glmstat(code,x[,model],y,b,bv,list(wx,weights,off,pow,nbk))
      if (crit=="bic")
        critval=stat.bic~stat.aic
      else
        critval=stat.aic~stat.bic
      endif
      if (shm)
        model=model'
        model
        if (crit=="bic")
          stat.bic
        else
          stat.aic
        endif
      endif
;      
      if (j==1) 
        if (exist(fix))   
          bestmod=critval~fix'~(imodel.*notfix')
        else
          bestmod=critval~(imodel.*notfix')
        endif
        bestmodord[1]=bestmod
      else
        if (exist(fix))   
          appmod=(critval~fix'~(imodel.*notfix'))
        else
          appmod=(critval~(imodel.*notfix'))
        endif
        bestmod=bestmod|appmod
        bestmod=sort(bestmod,1)
        bestmod=bestmod[1:min(5|rows(bestmod))]
        ord=sum(imodel,2)
        if (bestmodord[ord,1]==0)
          bestmodord[ord]=appmod
        endif
        if (critval[1,1] < bestmodord[ord,1])
          bestmodord[ord]=appmod
        endif
      endif
    endo
;
    bestmodel=bestmod[1,3:p+2]'
    bestmodel=paf(bestmodel,bestmodel!=0)
    {b,bv,it,ret}=glmcore(code,xr[,bestmodel],yr,wxr,offr,ctrl)
    stat=glmstat(code,x[,bestmodel],y,b,bv,list(wx,weights,off,pow,nbk))
;
    best=sort(fullmodel~bestmod[,3:p+2]')
    best=best[,2:cols(best)]
    if (crit=="bic")
      bic=bestmod[,1]'
      aic=bestmod[,2]'
    else
      aic=bestmod[,1]'
      bic=bestmod[,2]'
    endif
    bestcrit=list(aic,bic)
;
    bestord=sort(fullmodel~bestmodord[,3:p+2]')
    bestord=bestord[,2:cols(bestord)]
    if (crit=="bic")
      bic=bestmodord[,1]'
      aic=bestmodord[,2]'
    else
      aic=bestmodord[,1]'
      bic=bestmodord[,2]'
    endif
    bestordcrit=list(aic,bic)
  else
    ; ready!
    {b,bv,it,ret}=glmcore(code,xr,yr,wxr,offr,ctrl)
    stat=glmstat(code,x,y,b,bv,list(wx,weights,off,pow,nbk))
    if (shm)
      "'glmselect': nothing to select!"
    endif
    best=(1:p)
    aic=stat.aic
    bic=stat.bic
    bestcrit=list(aic,bic)
  endif
;  
  if (twoparfam)
    bv=stat.dispersion.*bv
  endif
  append(stat,it)
  append(stat,ret)
  append(stat,nr)
;
  bestfit=list(b,bv,stat)
endp







