proc(bp)=grbox(x, col)
; -----------------------------------------------------------------------
; Library      graphic
; -----------------------------------------------------------------------
; See_also     grbar grdot grhist grdotd grash
; -----------------------------------------------------------------------
; Macro        grbox
; -----------------------------------------------------------------------
; Description  Generates a boxplot with mean line and median line.
;              Outliers outside the intervall [Q_25-3*IQR, Q_75+3*IQR] 
;              will be plotted as crosses, outliers ouside the intervall
;              [Q_25-1.5*IQR, Q_75+1.5*IQR] will be plotted as circles.
; -----------------------------------------------------------------------
; Usage        bp = grbox (x {, col})
; Input
;   Parameter  x
;   Definition n x 1      vector
;   Parameter  col
;   Definition color
; Output
;   Parameter  bp
;   Definition composed graphical object
; -----------------------------------------------------------------------
; Example      library ("graphic")
;              t  = (normal(25)-8)|(normal(25)+8)|normal(100)|(normal(10)+25)
;              bp = grbox (t)
;              d  = createdisplay (1,1)
;              show (d, 1, 1, bp)
; -----------------------------------------------------------------------
; Result       shows a boxplot
; -----------------------------------------------------------------------
; Link         ../tutorials/graphicstart.html Introduction to the graphic library
; -----------------------------------------------------------------------
; Keywords     graphic primitives, boxplot
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 960925
; -----------------------------------------------------------------------
  error(cols(x).<>1, "grbox : cols(x)<>1")
  if (exist("col").<>1)
    grc = getglobal ("grc")  
    col = grc.col.black
  endif
  xs = sort(x)
  n1 = rows(x)+1
  if (abs( n1/2-floor(n1/2) )<0.1)  ; (n+1) is even
    dm  = floor(n1/2+0.5) 
    q50 = xs[dm]
  else			     ; (n+1) is odd
    dm  = floor(n1/2) 
    q50 = 0.5*(xs[dm]+xs[dm+1])
  endif
;;
;;  a = 0.25|0.75
;; q = xs[ceil (rows(x).*a) + (a==0)]
;;  q25 = q[1]
;;  q75 = q[2]
  d2 = dm+1
  if (abs( d2/2-floor(d2/2) )<0.1)  ; (dm+1) is even
    dm  = floor(d2/2+0.5)
    q25 = xs[floor(dm)]
    q75 = xs[floor(n1-dm)]
  else
    dm  = floor(d2/2) 
    q25 = 0.5*(xs[floor(dm)]+xs[dm+1])
    q75 = 0.5*(xs[floor(n1-dm)]+xs[n1-dm-1])
  endif  
;;
  mx = mean(xs)
  iqr = 1.5.*(q75-q25)
  lf = xs[1+sum(xs<q25-iqr)]
  uf = xs[sum(xs<=q75+iqr)]
  bp = (lf~0)|(lf~1)
  bp = bp|((q25~0)|(q25~1))
  bp = bp|((q50)~0|(q50~1))
  bp = bp|((q75)~0|(q75~1))
  bp = bp|((uf~0)|(uf~1))
  bp = bp|((mx~0)|(mx~1))
  bp = bp|((lf~0.5)|(q25~0.5)|(q75~0.5)|(uf~0.5)) 
  bll = #(1,3,5,7,9,11,13,15,3,5,4,6)~#(2,4,6,8,10,12,14,16,5,7,6,8)
  bal = #(1,1,1,1,1,3,1,1,1,1,1,1)
  bap = 3*matrix(16)
  bsp = 0*matrix(16)
  s = sum(xs<(q25-iqr))
  if (s)
    bt = paf(xs, xs<(q25-iqr))
    bp = bp|(bt~0.5.*matrix(rows(bt)))
   bap = bap|(3+2.*(bt<(q25-2.*iqr)))
   bsp = bsp|(8*matrix(rows(bt)))
  endif
  s = sum(xs>(q75+iqr))
  if (s)
    bt = paf(xs, xs>(q75+iqr))
    bp = bp|(bt~0.5.*matrix(rows(bt)))
    bap = bap|(3+2.*(bt>(q75+2.*iqr)))
    bsp = bsp|(8*matrix(rows(bt)))
  endif
  setmaskl (bp, bll, col, bal, 1*matrix(rows(bll)))
  setmaskp (bp, col, bap, bsp)
endp