proc (cil,kme,ciu)=hazkpm(data,alpha)   

; -----------------------------------------------------------------
; Library     hazreg
; -----------------------------------------------------------------
; See_also    hazdat, haznar, hazrisk
; -----------------------------------------------------------------
; Macro       hazkpm
; -----------------------------------------------------------------
; Description estimates the survival function at the sorted
;             observations of a right-censored data together
;             with the Greenwood pointwise confidence intervals.
; -----------------------------------------------------------------
; Usage       {cil, kme, ciu} = hazkpm(data{,alpha})
; Input       
; Parameter   data 
; Definition  n x (p+4) matrix, the first column is the sorted 
;             survival time t, followed by the sorted delta, 
;             inidcating if censoring has occured, labels l, a 
;             column containing the number of ties, and lastly, 
;             the sorted covariate matrix z.
; Parameter   alpha 
; Definition  scalar, the confidence level, optional, 
;             default = 0.05 
; Output
; Parameter   cil  
; Definition  n x 2 matrix, the first column is the sorted 
;             t, followed by the Greenwood lower confidence 
;             bound for the survival function at the points of t.
; Parameter   kme  
; Definition  n x 2 matrix, the first column is the sorted 
;             t, followed by the Kaplan-Meier estimates of the
;             survival function at the points of t.
; Parameter   ciu  
; Definition  n x 2 matrix, the first column is the sorted 
;             t, followed by the Greenwood upper confidence 
;             bound for the survival function at the points of t.
; -----------------------------------------------------------------
; Example     library("hazreg") 
;             
;             y = -log(1-uniform(20))         ; exponential survival
;             c = 5*uniform(20)               ; uniform censoring
;             t = min(y~c,2)                  ; censored time             
;             delta = (y<=c)                  ; censoring indicator            
;             
;             {data,ties} = hazdat(t,delta)   ; preparing data
;             {cil, kme, ciu} = hazkpm(data)  ; the Kaplan-Meier
;                                             ; estimates with the
;                                             ; Greenwood 
;                                             ; confidence bounds                              
; -----------------------------------------------------------------
; Result      The Kaplan-Meier estimates and Greenwood confidence 
;             bounds are obtained for the sorted censored data. 
; -----------------------------------------------------------------
; Example     library("hazreg") 
;             
;             y = 2|1|3|2|4|7|1|3|2        ; hypothetical survival
;             c = 3|1|5|6|1|6|2|4|5        ; hypothetical censoring
;             t = min(y~c,2)               ; censored time             
;             delta = (y<=c)               ; censoring indicator            
;             
;             {data,ties} = hazdat(t,delta)   ; preparing data  
;             {cil, kme, ciu} = hazkpm(data)  ; the Kaplan-Meier
;                                             ; estimates with the
;                                             ; Greenwood 
;                                             ; confidence bounds                              
; -----------------------------------------------------------------
; Result      The Kaplan-Meier estimates and Greenwood confidence 
;             bounds are obtained for the sorted censored data. 
;             There are ties in the data: three 1's, three 2's, 
;             two 3's. 
; -----------------------------------------------------------------
; Keywords    confidence interval, Kaplan-Meier Estimator
; -----------------------------------------------------------------
; Author      Lijian Yang, B. Grund 000308
; -----------------------------------------------------------------

;library("xplore")
  if (exist(alpha)==0)
    alpha = 0.05
  endif

  n=rows(data)

  t = data[,1]
	delta = data[,2]

  r = haznar(data)
  d = data[,4]          ; number of tied observed times

  nevents = matrix(n)   ; number of tied events
  i = 1
  while (i<=n)
    ti = t[i]
    diff = ti-t
    ind = (diff==0) .* delta
    nevents[i] = sum(ind)
    i = i + 1
  endo
 
  s = 1 - nevents./r
	v =  nevents./(r.*(r-nevents))

  d1 = d[1]             
  if (d1 > 1)           ; special case: tied obs. in t[1]
    j = 2
      while (j <= d1)
				s[j] = s[1]
				v[j] = v[1]
				j = j + 1
      endo
	endif
    

  i = d1 + 1            ; recursive computation of kpm, ciu, cil
  while (i<=n) 
    if (delta[i] == 1)
    	s[i] = (s[i])*(s[i-1])
  		v[i] = v[i] + v[i-1]
		else
    	s[i] = s[i-1]
      v[i] = v[i-1]
		endif
    di = d[i]
		if (di > 1)
			j = 1
      while (j < di)
				s[i+j] = s[i]
				v[i+j] = v[i]
				j = j + 1
      endo
		endif
    i = i + di
  endo

 
  v = sqrt(v)
  v = v.*s
  
  z = qfn(1-alpha/2)
  v = z.*v

 cil = s - v
 ciu = s + v

 
 i=1                 ; project CIs onto [0,1]
 while (i<=n)
   if (r[i] > nevents[i]) 
     cil[i] = max(0~cil[i],2) 
 		 ciu[i] = min(1~ciu[i],2) 
   else
     cil[i] = 0/0    ; no confidence limits when
     ciu[i] = 0/0    ; all observations are spent
   endif
   i=i+1
 endo

 cil = t~cil
 kme = t~s
 ciu = t~ciu
 
endp

