proc (surv) = hazsurv(data,z)   

; -----------------------------------------------------------------
; Library     hazreg
; -----------------------------------------------------------------
; See_also    hazdat, hazregll, hazbeta, hazbase
; -----------------------------------------------------------------
; Macro       hazsurv
; -----------------------------------------------------------------
; Description calculates the conditional survival function, 
;             using the maximum likelihood estimate of 
;             the regression parameter beta obatined through
;             hazbeta. 
; -----------------------------------------------------------------
; Usage       surv = hazsurv(data,z)
; Input       
; Parameter   data 
; Definition  n x (p+4) matrix, the first column is the sorted 
;             survival time t, followed by the sorted delta, 
;             inidcating if censoring has occured, labels l, a 
;             column containing the number of ties, and lastly, 
;             the sorted covariate matrix.
; Parameter   z 
; Definition  n x p matrix, the covariates.
; Output
; Parameter   surv  
; Definition  n x 2 matrix, the first column is the sorted 
;             t, followed by the estimated survival function 
;             at the points of t, conditional on z.
; -----------------------------------------------------------------
; Example  library("hazreg") 
;            n = 20
;            p = 2
;            beta = 1|2                      ; regression parameter
;            z = 1 + uniform(n,p)            ; covariates
;            y = -log(1-uniform(n))          ; exponential survival
;            y = y./exp(z*beta)              ; covariate effects
;            c = 4*uniform(n)                ; uniform censoring
;            t = min(y~c,2)                  ; censored time             
;            delta = (y<=c)                  ; censoring indicator            
;            
;         {data,ties} = hazdat(t,delta, z)   ; preparing data
;         z1 = 1.1|1.23
;         surv = hazsurv(data, z1)  
;                                            ; estimation of the
;                                            ; conditional survival
;                                            ; function                              
; -----------------------------------------------------------------
; Result     The conditional survival function is estimated. 
; -----------------------------------------------------------------
; Keywords   conditional survival function
; -----------------------------------------------------------------
; Author     Lijian Yang 990706
; -----------------------------------------------------------------
 
  {betahat,betak,ck} = hazbeta(data)
  {bhaz,bsurv} = hazbase(data)
  p = rows(betahat)
  t = data[,1]
  bsurv = bsurv[,2]

  zbeta = z'*betahat
  ezbeta = exp(zbeta)

  surv = (bsurv).^(ezbeta)
  
surv = t~surv
  
endp


