proc(m,b,const)=intestpl(x,t,y,h,g,loc,opt)
; -----------------------------------------------------------------
;   Library       gam
; -----------------------------------------------------------------
;   See_also      intest, gintest, gintestpl, gamfit, pcad
; -----------------------------------------------------------------
;   Macro         intestpl
; -----------------------------------------------------------------
;   Description   estimation of the univariate additive functions
;                 in a separable additive partial linear model
;                 using local polynomial estimation
; -----------------------------------------------------------------
;   Usage         {m,b,const} = intestpl(x,t,y,h,g,loc{,opt})
;   Input
;     Parameter   x
;     Definition      n x d  matrix, the discrete predictor variables.
;     Parameter   t
;     Definition      n x p  matrix, the continuous predictor variables.
;     Parameter   y
;     Definition      n x q  matrix , the observed response variables
;     Parameter   h
;     Definition      p x 1 or 1 x 1 matrix , chosen bandwidth for
;                     the directions of interest
;     Parameter   g
;     Definition      p x 1 or 1 x 1 matrix , chosen bandwidth
;                     for the directions not of interest
;     Parameter   loc
;     Definition      dummy , if loc=1 local linear, if loc=2 local
;                     quadratic estimator will be used,
;     Parameter   opt
;     Definition      optional, a list with optional input. The macro
;                     "gplmopt" can be used to set up this parameter.
;                     The order of the list elements is not important.
;                     Parameters which are not given are replaced by
;                     defaults (see below).
;     Parameter   opt.tg
;     Definition      ng x pg vector, a grid for continuous part. If tg is
;                     given, the nonparametric function will be
;                     computed on this grid.
;     Parameter   opt.shf
;     Definition      integer,  (show-how-far) if exists and =1, an output
;                     is produced which indicates how the iteration
;                     is going on (additive function / point of estimation /
;                     number of iteration).
;   Output
;     Parameter   m
;     Definition      ng x pp x q matrix,  where pp = p * loc,
;                     containing the marginal integration estimators
;                     of the additive components in the first p columns,
;                     the derivatives in the following.
;     Parameter   b
;     Definition      d x 1  matrix, containing the estimators of
;                     the linear factors of the corresponding
;                     linear influences
;     Parameter   const
;     Definition      scalar, containing the intercept
;
; -----------------------------------------------------------------
;   Example   library("gam")
;             randomize(1345)
;             loc= 2
;             x = matrix(50,2)
;             t = uniform(50,2)*2-1
;             xh = uniform(50,2)
;             x[,1]= 3*(xh>=0.8)+2*((0.8>xh)&&(xh>=0.3))+(0.3>xh)
;             x[,2]= (xh>(1/3))
;             g1    = 2*t[,1]
;             g2    = (2*t[,2])^2
;             g2    = g2 -mean(g2)
;             m     = g1 + g2 + x*(0.2|-1.0)
;             y     = m + normal(50,1)*0.25
;             h     = #(1.4, 1.4)
;             g     = #(1.4, 1.4)
;             {m,b,const} = intestpl(x,t,y,h,g,loc)
;             b
;             const
;             bild =createdisplay(1,2)
;             dat11= t[,1]~g1
;             dat12= t[,1]~m[,1]
;             setmaskp(dat12,4,4,8)
;             show(bild,1,1,dat11,dat12)
;             dat21= t[,2]~g2
;             dat22= t[,2]~m[,2]
;             setmaskp(dat22,4,4,8)
;             show(bild,1,2,dat21,dat22)
; -----------------------------------------------------------------
;   Result    the marginal integration estimates of the additive
;             functions and the vector of the linear part,
;             see Fan, Haerdle and Mammen (1996)
;             and the derivatives of the additive functions, using
;             local polynomials, see Severance-Lossin & Sperlich (1997).
; -----------------------------------------------------------------
;   Author    Sperlich & Stockmeyer 970711
; -----------------------------------------------------------------
;
; initialize
; -----------------------------------------------------------------
 if (comp("opt","shf")>0)
   shf = opt.shf
 else
   shf = 0
 endif
 d = cols(x)
 p = cols(t)
 n = rows(x)
 q = cols(y)
 o = sort(t[,1]~(1:n))[,2]
 ro = sort(o~(1:n))[,2]
 y = y[o]
 t = t[o]
 x = x[o]
 error(n<>rows(t),"dimensions of x and t do not fit")
 error(sum(loc==#(1,2))!=1,"loc is not correctly specified")
 loc = loc - 1                /* still needed for locpoldis */
 if (rows(g)<>p)              /* reshape g to matrix */
   errtext = "rows of g didn't fit: "+string("%1.4f",g[1,1])+" assumed"
   warning(rows(g)>1,errtext)
   g = matrix(p)*g[1,1]
 endif
 if (rows(h)<>p)            /* reshape h to matrix */
   errtext = "rows of h didn't fit: "+string("%1.4f",h[1,1])+" assumed"
   warning(rows(h)>1,errtext)
   h = matrix(p)*h[1,1]
 endif
;
; inner grid
; -----------------------------------------------------------------
 innergrid = (n>40)
 if (innergrid)
   nig = (n>499)*40+(n<500)*25+10
   tmin = min(t)
   tmax = max(t)
   range = tmax-tmin
   ig = tmin + (0:4)*(0.02*range)
   ig = ig | (tmin+range*0.1) + (0:nig-11)*(0.8*range/(nig-11))
   ig = ig | (tmin+range*23/25) + (0:4)*(0.02*range)
 else
   nig = n
   ig = t
 endif
 fhig = matrix(nig,(2+loc)*p,q)*0
 fh = matrix(n,(2+loc)*p,q)*0
 const = 0 * matrix(1,p,q)
;
; estimation routine
; -----------------------------------------------------------------
 j = 1
 while (j<=p)
   hv = g
   hv[j] = h[j]
   if (loc)
     posfh = j | (p+j) | (2*p+j)
     posmb = 1 | (1+j) | (1+p+j)
   else
     posfh = j | (p+j)
     posmb = 1 | (1+j)
   endif
   i = 1
   while (i<=nig)
     if (shf)
       ("intestpl: "+string("%1.0f",j)+" "+string("%1.0f",i))
     endif
     xest = t
     xest[,j] = ig[i,j].*matrix(n,1)
     ma = locpoldis(t,xest,y,hv,loc,x)
     k = 1
     while (k<=q)
       mb = ma[,,k]
       sel = isNumber(mb[,1])
       if (sum(sel,1)>10)
         mb = paf(mb,sel)
       endif
       fhig[i,posfh,k] = mean(mb[,posmb],1)
       k=k+1
     endo
     i = i + 1
   endo
   jpos = paf(j|p+j|2*p+j,1|(loc>=0)|(loc==1))
   if (innergrid)
     ig0 = ig[1:nig-1,j]
     ig1 = ig[2:nig,j]
     s = (t[,j]>=ig0')&&(t[,j]<ig1')
     f0 = fhig[1:nig-1,jpos]
     f1 = fhig[2:nig,jpos]
     fh[,jpos] = s*f0 + s*((f1-f0)/(ig1-ig0)) .* (t[,j]-s*ig0)
     upper = (t[,j]==tmax[,j])
     if (sum(upper)>0)
       fh[paf((upper.*(1:n)),upper),jpos] = fhig[nig,jpos]
     endif
   else
     fh[,jpos] = fhig[,jpos]
   endif
   sel = isNumber(fh[,j,1])
   if (sum(sel)>(0.5*n))
     const[1,j,] = mean(paf(fh[,1:p,],sel),1)
   endif
   j = j + 1
 endo
 if (loc)
   fh[,(2*p+1):(3*p),] = fh[,(2*p+1):(3*p),]*2
 endif
 fh[,1:p,] = fh[,1:p,] .- const
;
; estimation of b
; -----------------------------------------------------------------
 Ysum= y .- sum(fh[,1:p,],2)
; ols = matrix(n)~x[,(p+1):d] ??
 ols = matrix(n)~x
 b   = inv(ols'*ols)*ols' *Ysum
;
; prepare output / use grid tg
; -----------------------------------------------------------------
 if (comp("opt","tg")>0)
   ng = rows(opt.tg)
   tg = opt.tg
   fhg = matrix(ng,(2+loc)*p,q)*0
   j = 1
   while (j<=p)
     ig0 = ig[1:nig-1,j]
     ig1 = ig[2:nig,j]
     s = (tg[,j]>=ig0')&&(tg[,j]<ig1')
     jpos = paf(j|p+j|2*p+j,1|(loc>=0)|(loc==1))
     f0 = fhig[1:nig-1,jpos]
     f1 = fhig[2:nig,jpos]
     fhg[,jpos] = s*f0 + s*((f1-f0)/(ig1-ig0)) .* (tg[,j]-s*ig0)
     upper = (1:ng)'*(ig1[nig-1]==tg[,j])
     if (upper>0)
       fhg[upper,jpos] = fhig[nig,jpos]
     endif
   j = j + 1
   endo
   fhg[,1:p,] = fhg[,1:p,] .- const.*(fhg[,1:p,]!=0)
   m = fhg
 else
   k=1
   do
     fh[,,k] = fh[ro,,k]
     k=k+1
   until(k>q)
   m = fh
 endif
 const = b[1]
 b = b[2:d+1]
;
endp
