proc(Qn)=lo(y,trunclag)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  rvlm, kpss, robwhittle, roblm, lobrob, gph, sclmtest, neweywest 
;----------------------------------------------------------------------
;   Macro	  lo
; ----------------------------------------------------------------------
;   Description   Calculation of the Lo statistic for long-range dependence.
;                 The first argument of the quantlet is the series, the second 
;                 optional argument is the vector of truncation lags of the 
;                 autocorrelation consistent variance estimator. If the second
;                 optional argument is missing, the vector of truncation lags 
;                 is set to m = 5, 10, 25, 50. The quantlet returns the estimated 
;                 statistic with its corresponding order. If the estimated statistic 
;                 is outside the interval (0.809, 1.862), which is the 95 percent 
;                 confidence interval for no long-memory, 
;                 a star symbol * is displayed in the third column. The other 
;                 critical values are in Lo's paper.
;  
; ----------------------------------------------------------------------
;   Usage         q = lo(y{,m})
;   Input 
;     Parameter   y 
;     Definition  vector
;     Parameter   m
;     Definition  vector
;   Output
;     Parameter   q
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;detection of the existence of long-memory in the volatility
;               ;Since no optional argument is provided, Lo statistic is calculated
;               ;for the expansion orders 5, 10, 25, 50. Since for the orders 5 and 10,
;               ;the statistic is outside the 95 percent confidence interval for no 
;               ;long-memory, a star is displayed after the statistic.
;
;               library("times")
;               x = read("dmus58.dat")
;               x = x[1:2000]
;               y = abs(tdiff(x))
;               q = lo(y)
;               q
; ----------------------------------------------------------------------
;   Result
;               Contents of q
;
;              [1,] " Order   Statistic"
;              [2,] "__________________ "
;              [3,] ""
;              [4,] "    5     2.0012 *"
;              [5,] "   10     1.8741 *"
;              [6,] "   25     1.7490 "
;              [7,] "   50     1.6839 "
; 
; ----------------------------------------------------------------------
;   Example     ;detection of the existence of long-memory in the volatility
;               ;In that case, the Lo statistic is calculated for the expansion
;               ;orders 10, 15, and 20, which are provided as second arguments
;               ;of the quantlet. Since the statistic is outside the 95 percent 
;               ;confidence interval for no long-memory, for the order 10, a 
;               ;star is displayed after the statistic.
;
;               library("times")
;               x = read("dmus58.dat")
;               x = x[1:2000]
;               y = abs(tdiff(x))
;               m = #(10,15,20)
;               q = lo(y,m)
;               q
; ----------------------------------------------------------------------
;   Result
;               Contents of q

;

;              [1,] " Order   Statistic"
;              [2,] "__________________ "
;              [3,] ""
;              [4,] "   10     1.8741 *"
;              [5,] "   15     1.8065 "
;              [6,] "   20     1.7744 "
; 
; ----------------------------------------------------------------------
;   Reference
;      Lo, A.W. (1991): Long Term Memory in Stock Market Prices, 
;      Econometrica, 59, 1279-1313.
; ----------------------------------------------------------------------
;   Keywords
;      long memory, tests
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980425. Revised 981210
; ----------------------------------------------------------------------
  opt = 0 ; tests the presence of the optional matrix of truncation lags parameters
  if (exist(trunclag)==1)
    nbtrunclag = rows(trunclag)
    q = rint(trunclag)
	else
    nbtrunclag = 4
    q = #(5,10,25,50) 
  endif
  ;;
  n = rows(y)
  sy = cumsum(y-mean(y))
  numer = max(sy)- min(sy)
  ;; 
  Qn = matrix(nbtrunclag)*numer
  CI = matrix(nbtrunclag)
	sigma = sqrt(neweywest(y,q)*n)
  Qn = Qn/sigma
  ;;
  i=1
  while (i <= nbtrunclag)
    if ((Qn[i] >= 0.809) && (Qn[i] <= 1.862))
			CI[i] = 0
		else
			CI[i] = 42
		endif
    i = i+1
  endo  
  ;; display of the results
  line1 =" Order   Statistic"
  line2 ="__________________ "
  st1 = string("%5.0f",q)
  st2 = string("%11.4f",Qn)
	st2s= char(matrix(nbtrunclag,2)*32)
	st3 = char(CI)
  st4 = st1+st2+st2s+st3
  Qn = line1|line2|""|st4
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro lo
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

