proc(dp)=plotstar(x, option, col)
; -----------------------------------------------------------------------
; Library      plot
; -----------------------------------------------------------------------
; See_also     plotpcp plotandrews plotsplom plotgt
; -----------------------------------------------------------------------
; Macro        plotstar
; -----------------------------------------------------------------------
; Description  Plots a star diagram.
; -----------------------------------------------------------------------
; Usage        bp = grstar (x {, option {, col}})
; Input
;   Parameter  x 
;   Definition n x 1      vector
;   Parameter  option
;   Definition scalar     
;   Parameter  col
;   Definition color
; Output
;   Parameter  dp
;   Definition composed graphical object
; -----------------------------------------------------------------------
; Notes        option[1,1] allows some transformation on x:
;              grc.prep.none       0     no transformation    (default)
;              grc.prep.standard   1     standardize
;              grc.prep.zeroone    2     to [0,1]
;              grc.prep.pcacov     3     pca on cov matrix    
;              grc.prep.pcacorr    4     pca on corr matrix
;              grc.prep.sphere     5     sphering
; 
;              option[2,1] decides about reference lines:
;              grc.star.none       0     no reference         (default)
;              grc.star.standard   1     reference mean / std. dev
;              grc.star.robust     2     reference median / iqr
;
;              option[3,1] fixes the number of stars in y direction.
; -----------------------------------------------------------------------
; Example      ; loads the library plot
;              library ("plot")
;              ; reads the swiss banknote data (200x6)
;              x = read("bank2")
;              ; no transformation of the data, no reference lines and
;              ; 20 star diagram in the vertical direction
;              opt = 0|0|20
;              ; show the data as star diagrams
;              plotstar (x, opt)
; -----------------------------------------------------------------------
; Result       shows star diagrams
; -----------------------------------------------------------------------
; Link         ../tutorials/graphicstart.html Introduction to the graphic library
; -----------------------------------------------------------------------
; Keywords    high level graphics, star diagram
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 970418
; -----------------------------------------------------------------------
   error(cols(x).<3, "grstar : cols(x)<3")
   grc = getglobal ("grc")  
   if (exist("col").<>1)
     col = grc.col.black
   endif
   n   = rows(x)
   if (rows(col)==1)
     col = col.*matrix(n, cols(col))
   endif
   if (exist("option").<>1)
     option = grc.star.none|grc.prep.none|ceil(sqrt(n))
   endif
   if (rows(option)<3)
     if (rows(option)<2)
       option = option|grc.star.none
     endif
     option = option|floor(sqrt(n))
   endif
   y = transform (x, option[1])
   yminmax = min(y)|max(y)
   if (option[2]==grc.star.standard)
     yminmax = min(min(y)|(mean(y)-1.96*sqrt(var(y))))
     yminmax = yminmax|(max(y)|(mean(y)+1.96*sqrt(var(y))))
   endif
   sm = grstar (y, col, option[3], yminmax)
   grdisp = createdisplay(1,1)
   switch 
   case (option[2]==grc.star.none)
     show (grdisp, 1, 1, sm)
     break
   case (option[2]==grc.star.standard)
     r  = (mean(y)-1.96.*sqrt(var(y))).*matrix(n)
     sl = grstar (r, col, option[3]|2, yminmax)
     r  = (mean(y)+1.96.*sqrt(var(y))).*matrix(n)
     su = grstar (r, col, option[3]|2, yminmax)
     show (grdisp, 1, 1, su, sm, sl)
     break
   case (option[2]==grc.star.robust)
     r = quantile(y, 0.25).*matrix(n)
     sl = grstar (r, col, option[3]|2, yminmax)
     r = quantile(y, 0.75).*matrix(n)
     su = grstar (r, col, option[3]|2, yminmax)
     show (grdisp, 1, 1, su, sm, sl)
     break
   endsw
endp