proc (r) = qfx (dist, q, shape)
; -----------------------------------------------------------------
;   Library       stats
; -----------------------------------------------------------------
;   See_also      pdfx cdfx randx 
; -----------------------------------------------------------------
;   Macro         qfx
; -----------------------------------------------------------------
;   Description   evaluates extreme value and generalized Pareto
;                 quantile function for elements of a vector.
; -----------------------------------------------------------------
;   Usage         r = qfx (dist, q, shape)
;   Input
;     Parameter   dist
;     Definition      name of distribution: "ev", "ev0", "ev1", "ev2", "gp", "gp0", "gp1" or "gp2"
;     Parameter   x
;     Definition      vector
;     Parameter   shape
;     Definition      shape parameter
;   Output
;     Parameter   r
;     Definition      vector
; -----------------------------------------------------------------
;   Example   library("finance")
;             qfx("ev0", 0.5)
; -----------------------------------------------------------------
;   Result    0.36651
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------

error (sum (q < 0 || q > 1) > 0, "Invalid argument in qfx")
if (dist == "ev0")
  r = -log (-log (q))
endif
if (dist == "ev1")
  r = (-log (q))^(-1 / shape)
endif
if (dist == "ev2")
  r = -(-log (q))^(-1/shape)
endif
if (dist == "ev")
  if (abs (shape) < 1e-10)
    r = qfx ("ev0", q)
  else
    r = ((-log(q))^(-shape)-1)/shape
  endif
endif
if (dist == "gp0")
  r = -(log(1-q))
endif
if (dist == "gp1")
  r = (1-q)^(-1/shape)
endif
if (dist == "gp2")
  r = -(1-q)^(-1/shape)
endif
if (dist == "gp")
  if (abs (shape) > 1e-10)
    r = ((1 - q)^(-shape) - 1) / shape
  else
    r = qfx ("gp0", q)
  endif
endif

error (exist ("r") <= 0, "Distribution " + dist + " unknown")
endp
