proc (mh)=regest(x,h,K,d)   
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regci regcb regxest regxbwsel lpregest regestp 
; -----------------------------------------------------------------
;   Macro        regest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator
; -----------------------------------------------------------------
;   Description  computes the Nadaraya-Watson estimator for 
;                univariate regression. The computation uses WARPing.
; -----------------------------------------------------------------
;   Notes        The WARPing enhances the speed of computation,
;                but may lead to computational errors, if the
;                bandwidth is small. For exact computation, the
;                macro "regxest" should be used instead.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
; -----------------------------------------------------------------
;   Usage        mh = regest(x {,h {,K} {,d} })
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, 20% of the
;                  range of x[,1] is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and 
;                  (max(x[,1])-min(x[,1]))/100 is used. 
;   Output
;     Parameter  mh  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the regression estimate on 
;                  that grid.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = 4.*pi.*(uniform(200)-0.5)   ; independent variable
;             m = cos(x)                      ; true function
;             e = uniform(200)-0.5            ; error term             
;             x = x~(m+e)                             
;             ;
;             mh = regest(x,1)                ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             m  = setmask(sort(x[,1]~m) , "line","black","thin")
;             plot(x,mh,m)                                         
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimate (blue) using   
;             Quartic kernel and bandwidth h=1 and the true 
;             regression function (thin black) are pictured.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = read("motcyc")            ; read motorcycle data
;             mhe = regest(x,3,"epa")       ; estimate function
;             mhu = regest(x,2,"uni")       ; estimate function
;             ;
;             mhe= setmask(mhe,"line","green")
;             mhu= setmask(mhu,"line","red")
;             plot(mhe,mhu)                 ; graph functions
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimates using   
;             Epanechnikov kernel (green) and Uniform kernel 
;             (red) are pictured.  
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/03/28 - 14:04
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"regest: cols(x) <> 2")
;
  if (exist(h)==0)
    h=(max(x[,1])-min(x[,1]))/5
  endif
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))/100
    d=min(d~(h./3),2)
  endif
;
  error(h .<=d,"denest: h smaller than d -- Please modify h or d!")
;
  {xb,yb}=bindata(x[,1],d,0,x[,2])   ; bin data in x and sum of y's   
  if (K!="gau")
    wy=symweigh(0,d/h,h/d,K)         ; create weights for quartic kernel
  else
    wy=symweigh(0,d/h,5.*h/d,K)        
  endif    
  wx=aseq(0,rows(wy),1)
  {xc,yc,or}=conv(xb,yb,wx,wy)       ; smooth x's and y's   
  mh=(d.*xc+d.*0.5)~(yc[,2]./yc[,1]) 
;  mh=paf(mh,or!=0)
endp


