proc(mh, clo, cup) = regxcb(x,h,alpha,K,xv) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regxest regxci regest regci regcb 
; -----------------------------------------------------------------
;   Macro        regxcb
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator, confidence bands
; -----------------------------------------------------------------
;   Description  computes uniform confidence bands with 
;                prespecified confidence level for univariate
;                regression using the Nadaraya-Watson estimator.
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                exploratory purposes, the macro "regcb" is
;                recommended, which uses the faster WARPing method.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        {mh, clo, cup} = regxcb(x {,h {,alpha {,K} {,d}}})
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, 20% of the
;                  range of x[,1] is used.
;     Parameter  alpha  
;     Definition   confidence level, If not given, 0.05 is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] with
;                  K(-1)=K(1)=0. If not given, the Quartic kernel 
;                  "qua" is used.
;     Parameter  xv  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  x is used.
;   Output
;     Parameter  mh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the 
;                  second column contains the regression estimate  
;                  on the values of the first column.
;     Parameter  clo  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the
;                  second column contains the lower confidence 
;                  bounds on the values of the first column.
;     Parameter  cup  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the
;                  second column contains the upper confidence 
;                  bounds on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")                                    
;             x = read("motcyc")
;             {mh, clo, cup} = regxcb(x,3)  
;             ;
;             x =setmask( x,"cross")
;             mh=setmask(mh,"line","blue")
;             clo=setmask(clo,"line","blue","thin","dashed")
;             cup=setmask(cup,"line","blue","thin","dashed")
;             plot(x,mh,clo,cup)
; -----------------------------------------------------------------
;   Result    Uniform confidence bands at confidence level 
;             alpha=0.05 are pictured using the bandwidth h=0.3
; -----------------------------------------------------------------
;   Author    Marlene Mueller 990413
; -----------------------------------------------------------------
;
  error(cols(x)<>2,"regxcb: cols(x) <> 2")
; 
  if (exist(h)==0)
    h=(max(x[,1])-min(x[,1]))/5
  endif
  if (exist(alpha)==0)
   alpha = 0.05                 ; default confidence level is 0.05    
  endif   
  if (exist(K)==0)
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if (exist(K)==1) ; no K but xv
    xv=K
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if ((exist(K)==2)&&(K!="qua"))
    {dK,cK,mK}=canbw(K)
    {dK,cK,mK}=canbw(K)
    ng=1000
    hg=2/(ng-1)
    g=grid(-1,hg,ng)
    g1=g[1:(rows(g)-1)]
    g2=g[2:rows(g)]
    DK=(_K(g2)-_K(g1))./hg  ; approx. derivative of K
    c1K=sum(DK^2)*hg        ; approx. norm of K'
    c2=c1K/(2*cK)
  endif
;
  tmp=sort(x)
  x=tmp[,1]
  y=tmp[,2]
  if (exist(xv)==0)
    xv=x
  endif
;
  n  = rows(x)
  one=matrix(n)
;
  n = rows(x)
  mx= min(x[,1])
  rx= max(x[,1])-min(x[,1])
;
  x[,1] = (x[,1]-mx)./rx
  xv = (xv-mx)./rx
  h = h./rx
;
  if (K!="gau")
    tmp=sker(x,h,K,one~y,xv)
  else
    tmp=sker(x,5.*h,"ngau",one~y,xv)
  endif
  mh=xv~ (tmp[,2]./tmp[,1])
;
  if (K!="gau")
    tmp=sker(x,h,K,one~(y^2),xv)
  else
    tmp=sker(x,5.*h,"ngau",one~(y^2),xv)
  endif
  sh = (tmp[,2]./tmp[,1])
  sh = sh - mh[,2].^2     ; estimate for var(x[,2]|x[,1])
  fh = tmp[,1]
;
  r  = sqrt(-2*log(h))
  dn = r + 0.5*log( c2/(2*(pi^2)) )/r
  calpha = -log(-log(1-alpha)/2)
;
  mrg = (calpha/r + dn) .* sqrt(cK.*sh./((n*h).*fh))
  clo = (rx.*mh[,1] + mx)~(mh[,2] - mrg)
  cup = (rx.*mh[,1] + mx)~(mh[,2] + mrg)
  mh  = (rx.*mh[,1] + mx)~(mh[,2])
endp  


