proc (xr, yr, ro) = replicdata (x, y)
; ----------------------------------------------------------------------------
; Library       gplm
; ----------------------------------------------------------------------------
;  See_also     discrete sort cumsum paf
; ----------------------------------------------------------------------------
;   Macro       replicdata
; ----------------------------------------------------------------------------
;   Description  replicdata reduces a matrix x to its distinct 
;                rows and gives the number of replications of
;                each rows in the original dataset. An optional
;                second matrix y can be given, the rows of y
;                are summed up accordingly.
;                replicdata does in fact the same as discrete
;                but provides an additional index vector to
;                identify the reduced data with the original.
;                It takes a little longer due an additonal sort.
; ----------------------------------------------------------------------------
;   Usage        {xr,yr,ro} = replicdata(x{,y})
;   Input
;     Parameter   x
;     Definition        n x p matrix, the data matrix to reduce, in 
;                       regression usually the design matrix.
;     Parameter   y
;     Definition        optional , n x q matrix, in regression 
;                       usually the observations of the dependent variable.
;   Output
;     Parameter   xr
;     Definition        m x p matrix, reduced data matrix (sorted).
;     Parameter   yr
;     Definition        m x 1 vector or m x (q+1) matrix, contains in the first
;                       column the number of replications. If y was given, sums
;                       of y-rows with same x-row are contained in the other
;                       q columns of r.
;     Parameter   ro
;     Definition        n x 1 vector, index vector to rearrange estimators
;                       obtained by xr, yr according to the original 
;                       observations.
; ----------------------------------------------------------------------------
;   Example   library("gplm")
;             n=100
;             x=2*sort(ceil(10*uniform(n))./10)-1
;             y=cos(pi*x) + normal(n)
;             ; --------------------------------------
;             ;  data reduction
;             ; --------------------------------------
;             {xr,yr,o}=replicdata(x,y)
;             r =yr[,1]
;             yr=yr[,2]
;             rows(r)
;             ; --------------------------------------
;             ;  kernel regression of yr on xr
;             ; --------------------------------------
;             sr =sker(xr,1,"qua",r~yr)
;             mhr=sr[,2]./sr[,1]
;             ; --------------------------------------
;             ;  get prediction for all obs. from mhr
;             ; --------------------------------------
;             mh=mhr[o]
;             ; --------------------------------------
;             ;  compare
;             ; --------------------------------------
;             s  = sker(x,1,"qua",matrix(n)~y)
;             mc = s[,2]./s[,1]
;             sum(abs(mc-mh))
; ----------------------------------------------------------------------------
;   Result    Matrices x, y with 100 rows are reduced to a matrix xr 
;             (containing distinct rows of x) and yr (sums of y with same 
;             rows in x). 
;             r gives the number of replications. 
;             The regression of y on x coincides with the weighted 
;             regression of yr on xr.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/10
; ----------------------------------------------------------------------------
  n = rows (x)
  p = cols(x)
  ro= 1:n
  havey= exist(y)==1
;
  if (havey)
    x = sort (x~y~ro,(1:p))
    xo= x[,cols(x)]
    y = cumsum(x[,p+1:cols(x)-1])
    x = x[,1:p]
  else
    x = sort (x~ro,(1:p))
    xo= x[,cols(x)]
    x = x[,1:p]
  endif
  d = sum(abs(x[2:n]-x[1:n-1]),2) != 0
  ro = cumsum(0|d)+1
  if (sum(d)>0)
    s = 0| paf (1:(n-1), d) |n
    ns = rows(s)
    xr = x[1+s[1:ns-1]]
    yr = (s[2:ns]-s[1:ns-1])
    if (havey)
      yn = y[n]
      y0 = 0.*matrix(1,cols(y))
      y = y0| paf(y[1:n-1],d)| yn
      yr=yr~(y[2:ns]-y[1:ns-1])
    endif
  else
    xr = x[1]
    yr = n
    if (havey)
      yr= yr~y[n]
    endif
  endif
;
  ro = sort(xo~ro)[,2]  
endp
