; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   Macro         xtremes
; -----------------------------------------------------------------
;   Description   Graphical user interface for extreme value (EV) and
;                 generalized Pareto (GP) estimators
; -----------------------------------------------------------------
;   Usage         xtremes (x)
;   Input
;     Parameter   x
;     Definition      vector
; -----------------------------------------------------------------
;   Example   library("finance")
;             xtremes (randx("gp", 100, 1))
; -----------------------------------------------------------------
;   Result    Interactive analysis of data set
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------

proc () = xtremes (x)
  x = sort (x)
  str = " Estimator             k      shape         mu      sigma"
  str
  do
    do
      j = selectitem ("Select estimator", "Hill (GP1)" | "Pickands (GP)" | "Drees-Pickands (GP)" | "Moment (GP)" | "MLE (GP)" | "MLE (GP0)" | "Empirical Mean Excess" | "Empirical DF" | "Empirical Quantile Function" | "Kernel Density" | "Quit", "single")
      if (sum (j) == 1)
        i = paf (1:11, j)
      endif
    until (sum (j) == 1)
    estname = ""
    esttitle = ""
    switch
      case (i == 1)
        estname = "hillgp1"
        esttitle = "Hill (GP1)         "
        break
      case (i == 2)
        estname = "pickandsgp"
        esttitle = "Pickands (GP)      "
        break
      case (i == 3)
        estname = "dpgp"
        esttitle = "Drees-Pickands (GP)"
        break
      case (i == 4)
        estname = "momentgp"
        esttitle = "Moment (GP)        "
        break
      case (i == 5)
        estname = "mlegp"
        esttitle = "MLE (GP)           "
        break
      case (i == 6)
        estname = "mlegp0"
        esttitle = "MLE (GP0)          "
        break;
      case (i == 7)
        y = sort (x) [1:rows(x)-1]
        if (exist ("meanexc") != 4)
          meanexc = createdisplay (1, 1)
        endif
        adddata (meanexc, 1, 1, xtrplotlines (y, empme (x, y)))
        break
      case (i == 8)
        if (exist ("df") != 4)
          df = createdisplay (1, 1)
        endif
        data = xtrplotlines (x, (1:rows (x)) / rows (x))
        adddata (df, 1, 1, data)
        break
      case (i == 9)
        if (exist ("qf") != 4)
          qf = createdisplay (1, 1)
        endif
        data = xtrplotlines ((1:rows (x)) / rows (x), x)
        adddata (qf, 1, 1, data)
        break
      case (i == 10)
        if (exist ("density") != 4)
          density = createdisplay (1, 1)
        endif
        data = setmask (denest (x), "line", "black", "thin", "solid")
        adddata (density, 1, 1, data)
        break
    endsw
    if (estname != "")
     do
      do
        k = readvalue ("Number of  Extr. (0 for diagram except MLE(GP0))", 0)
        if (k == 0 && estname != "mlegp0")
          d = estname+"diag"
          m = _d(x, 8:rows (x))
          if (exist ("diag") != 4)
            diag = createdisplay (1, 1)
          endif
          adddata (diag, 1, 1, xtrplotlines ((rows (x) - rows (m) + 1):rows (x), m))
        endif
      until (k <> 0)
      m = _estname (x, k)
      if (estname == "mlegp0")
        gamma = 0
        append (m,gamma)
      endif
      if (estname == "hillgp1")
        str = esttitle + string ("%5g %10g %10g %10g (GP1 parametrization)", k, m.alpha, 0, m.sigma)
      else
        str = esttitle + string ("%5g %10g %10g %10g", k, m.gamma, m.mu, m.sigma)
      endif
      str
      do
        do
          sel = selectitem (esttitle, "Return to Estimator" | "Mean Excess Function" | "Distribution Function" | "Quantile Function" | "Density" | "QQ-Plot" | "Main Menu", "single")
          if (sum (sel) == 1)
            j = paf (1:7, sel)
          endif
        until (sum (sel) == 1)
        switch
          case (j == 2) // mean excess
            if (estname == "hillgp1")
              if (m.alpha >= 1)
                "Mean excess function does not exist"
                x0 = m.sigma
                x1 = max (x)
                y0 = m.sigma * gp1me (m.alpha, x0 / m.sigma)
                y1 = m.sigma * gp1me (m.alpha, x1 / m.sigma)
                if (exist ("meanexc") != 4)
                  meanexc = createdisplay (1, 1)
                endif
                adddata (meanexc, 1, 1, xtrplotlines (#(x0,x1),#(y0,y1)))
              endif
            else
              if (m.gamma >= 1)
                "Mean excess function does not exist"
              else
                x0 = m.mu + 0.001 * abs (m.mu)
                if (m.gamma >= 0)
                  x1 = max (x)
                else
                  x1 = m.mu - m.sigma / m.gamma 
                  x1 = x1 - 0.001 * abs (x1)
                endif
                y0 = m.sigma * gpme (m.gamma, (x0 - m.mu) / m.sigma)
                y1 = m.sigma * gpme (m.gamma, (x1 - m.mu) / m.sigma)
                if (exist ("meanexc") != 4)
                  meanexc = createdisplay (1, 1)
                endif
                adddata (meanexc, 1, 1, xtrplotlines (#(x0,x1),#(y0,y1)))
              endif
            endif
            break
          case (j == 3) // df
            if (estname == "hillgp1")
              x0 = m.sigma
              x1 = m.sigma * qfx ("gp1", 0.995, m.alpha)
              t = aseq (x0, 100, (x1 - x0) / 100)
              u = cdfx ("gp1", t / m.sigma, m.alpha)
            else
              x0 = m.mu
              x1 = m.mu + m.sigma * qfx ("gp", 0.995, m.gamma)
              t = aseq (x0, 100, (x1 - x0) / 100)
              u = cdfx ("gp", (t - m.mu) / m.sigma, m.gamma)
            endif
            if (exist ("df") != 4)
              df = createdisplay (1, 1)
            endif
            data = xtrplotlines (t, u)
            adddata (df, 1, 1, data)
            break
          case (j == 4) // qf
            t = aseq (0, 101, 0.01)
            t [101] = 1
            if (estname == "hillgp1")
              u = m.sigma * qfx ("gp1", t, m.alpha)
            else
              u = m.mu + m.sigma * qfx ("gp", t, m.gamma)
            endif
            if (exist ("qf") != 4)
              qf = createdisplay (1, 1)
            endif
            data = xtrplotlines (t, u)
            adddata (qf, 1, 1, data)
            break
          case (j == 5) // density
            if (estname == "hillgp1")
              x0 = m.sigma + m.sigma / 1000000
              x1 = m.sigma * qfx ("gp1", 0.995, m.alpha)
              t = aseq (x0, 500, (x1 - x0) / 500)
              u = pdfx ("gp1", t / m.sigma, m.alpha) / m.sigma
            else
              x0 = m.mu + m.sigma / 1000000
              x1 = m.mu + m.sigma * qfx ("gp", 0.995, m.gamma)
              t = aseq (x0, 500, (x1 - x0) / 500)
              u = pdfx ("gp", (t - m.mu) / m.sigma, m.gamma) / m.sigma
            endif
            if (exist ("density") != 4)
              density = createdisplay (1, 1)
            endif
            data = xtrplotlines (t, u)
            adddata (density, 1, 1, data)
            break
          case (j == 6) // QQ-plot
            if (estname == "hillgp1")
              t = m.sigma * qfx ("gp1", (1:rows (x)) / (rows (x)+1), m.alpha)
            else
              t = m.mu + m.sigma * qfx ("gp", (1:rows (x)) / (rows(x)+1), m.gamma)
            endif
            u = x 
            if (exist ("qqplot") != 4)
              qqplot = createdisplay (1, 1)
            endif
            data = xtrplotlines (t, u)
            adddata (qqplot, 1, 1, data)
            break           
        endsw
      until (j == 1 || j == 7)
     until (j == 7)
    endif
  until (i == 11)       
endp

proc (r) = xtrplotlines (x, y)
  if (existglobal ("xtrcolnum"))
    xtrcolnum = getglobal ("xtrcolnum")
  else
    xtrcolnum = 0
  endif  
  r = x~y
  switch
    case (xtrcolnum == 0)
      col = "black"
      break
    case (xtrcolnum == 1)
      col = "blue"
      break
    case (xtrcolnum == 2)
      col = "green"
      break
    case (xtrcolnum == 3)
      col = "cyan"
      break
    case (xtrcolnum == 4)
      col = "red"
      break
    case (xtrcolnum == 5)
      col = "magenta"
      break
  endsw
  r = setmask (r, "line", col, "thin", "solid")
  xtrcolnum = xtrcolnum + 1
  if (xtrcolnum == 6)
    xtrcolnum = 0
  endif
  putglobal ("xtrcolnum")
endp

