proc (res) = SPKRsurfgls (np, covmod, xmat, nx, dval, alpha, se, D)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRexpcov SPKRgaucov SPKRsphercov
;              SPKRtrmat SPKRprmat SPKRsemat SPKRcorrelogram SPKRvariogram
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRsurfgls
; -----------------------------------------------------------------------
; Description  Fits a trend surface by generalized least squares.
; -----------------------------------------------------------------------
; Usage        myres = SPKRsurfgls (np, covmod, xmat, nx, dval, alpha, se {, D})
; Input
;   Parameter  np
;   Definition scalar -  degree of the polynomial surface, an integer 
;                       in the range 0..6
;   Parameter  covmod
;   Definition integer - type of spatial covariance function:
;                       0 = expcov, 1 = gaucov, 2 = sphercov
;   Parameter  xmat
;   Definition n x 3  -  matrix of locations (x_i, y_i) [columns 1 & 2] and 
;                       observations z_i [column 3]
;   Parameter  nx
;   Definition integer - number of bins used for table of the covariance;
;                       increasing nx adds accuracy, but also increases
;                       the size of the object
;   Parameter  dval
;   Definition scalar -  range parameter (for spatial covariance function)
;   Parameter  alpha
;   Definition sclar  -  proportion of nugget effect (for spatial covariance 
;                       function)
;   Parameter  se
;   Definition scalar -  standard deviation at distance zero (for spatial 
;                       covariance function)
;   Parameter  D
;   Definition scalar -  optional; default D = 2; dimension of spheres 
;                       (only used for spatial covariance function SPKRsphercov)
; Output
;   Parameter  res
;   Definition list  -   consisting of components 
;                       x, y, z, np, f, alph, l, r, beta, wz, yy, W, l1f, 
;                       minx, maxx, covmod, type --
;   Parameter  x
;   Definition n x 1  -  same as xmat[,1] 
;   Parameter  y
;   Definition n x 1  -  same as xmat[,2] 
;   Parameter  z
;   Definition n x 1  -  same as xmat[,3] 
;   Parameter  np
;   Definition scalar -  same as input value np
;   Parameter  f
;   Definition matrix -  internal use only
;   Parameter  alph
;   Definition matrix -  internal use only
;   Parameter  l
;   Definition matrix -  internal use only
;   Parameter  r
;   Definition matrix -  internal use only
;   Parameter  beta
;   Definition matrix -  (np + 1)(np + 2) / 2 coefficients
;   Parameter  wz
;   Definition matrix -  internal use only
;   Parameter  yy
;   Definition matrix -  internal use only
;   Parameter  W
;   Definition matrix -  internal use only
;   Parameter  l1f
;   Definition matrix -  internal use only
;   Parameter  minx
;   Definition 1 x 3  -  minimum of columns of xmat
;   Parameter  maxx
;   Definition 1 x 3  -  maximum of columns of xmat
;   Parameter  covmod
;   Definition scalar -  same as input value covmod
;   Parameter  type
;   Definition string -  "trgls"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       topo = read("topo.dat")
;	       ; calculate a trend surface of order 2, using expcov as
;              ; spatial covariance function with dval = 0.7, alph = 0,
;              ; and se = 1
;  	       myres = SPKRsurfgls(2, 0, topo, 1000, 0.7, 0, 1)
; -----------------------------------------------------------------------
; Result       A list consisting of input parameters, intermediate
;              results, and final results of a trend surface by generalized 
;              least squares. This list will be used in other spatial
;              statistics quantlets such as SPKRtrmat, SPKRcorrelogram,
;              or SPKRvariogram.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("np") != 1, "np must be numeric")
  error ((np < 0) || (np > 6), "np must be >= 0 and <= 6")
  error (exist("covmod") != 1, "covmod must be numeric")
  error ((covmod != 0) && (covmod != 1) && (covmod != 2), "covmod must be 0, 1, or 2")
  error (exist("xmat") != 1, "xmat must be a matrix")
  error (cols(xmat) < 3, "xmat must have at least 3 columns")
  error (exist("nx") != 1, "nx must be numeric")
  error (nx < 1, "nx must be >= 1")
  error (exist("dval") != 1, "dval must be numeric")
  error (dval <= 0, "dval must be > 0")
  error (exist("alpha") != 1, "alpha must be numeric")
  error ((alpha < 0) || (alpha > 1), "alpha must be >= 0 and <= 1")
  error (exist("se") != 1, "se must be numeric")
  error (se < 0, "se must be >= 0")
  if (exist("D") == 0) ; parameter D has not been assigned
    D = 2
  endif
  ;
  ;
  x = xmat[,1]
  y = xmat[,2]
  z = xmat[,3]
  minx = min(xmat)
  maxx = max(xmat)
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall("_VR_frset", minx[1,1], maxx[1,1], minx[1,2], maxx[1,2])
  else  
    ; Unix
    d = dlcall("VR_frset", minx[1,1], maxx[1,1], minx[1,2], maxx[1,2])
  endif
  ;
  mm = 1.5 * sqrt((maxx[1,1] - minx[1,1])^2 + (maxx[1,2] - minx[1,2])^2)
  switch
    case (covmod == 0) ; expcov
      expcov = SPKRexpcov (grid(0, mm / nx, nx+1), dval, alpha, se)
      alph = (mm / nx) | expcov
      break
    case (covmod == 1) ; gaucov
      gaucov = SPKRgaucov (grid(0, mm / nx, nx+1), dval, alpha, se)
      alph = (mm / nx) | gaucov
      break
    case (covmod == 2) ; sphercov
      sphercov = SPKRsphercov (grid(0, mm / nx, nx+1), dval, alpha, se, D)
      alph = (mm / nx) | sphercov
      break
    default
      break
  endsw
  ;
  nalph = rows(alph)
  ;
  if (os == "windows")
    d = dlcall("_VR_alset", alph, nalph)
  else  
    ; Unix
    d = dlcall("VR_alset", alph, nalph)
  endif
  ;
  n = rows(xmat)
  npar = ((np + 1) * (np + 2)) / 2
  f = 1:(n * npar)
  ;
  if (os == "windows")
    d = dlcall("_VR_fmat", f, x, y, n, np)
  else  
    ; Unix
    d = dlcall("VR_fmat", f, x, y, n, np)
  endif
  ;
  l = 1:((n * (n+1)) / 2)
  r = 1:((npar * (npar+1)) / 2)
  beta = 1:npar
  wz = 1:n
  yy = 1:n
  W = 1:n
  ifail = 0
  l1f = 1:(n * npar)
  intvalues = n | np | npar | ifail
  ;
  if (os == "windows")
    d = dlcall("_VR_gls", x, y, z, f, l, r, beta, wz, yy, W, l1f, intvalues)
  else  
    ; Unix
    d = dlcall("VR_gls", x, y, z, f, l, r, beta, wz, yy, W, l1f, intvalues)
  endif
  ;
  ifail = intvalues[4]
  error (ifail > 0, "Rank failure in Choleski decomposition")
  type = "trgls"
  res = list (x, y, z, np, f, alph, l, r, beta, wz, yy, W, l1f, minx, maxx, covmod, type)
endp
