proc (res) = SPKRtrmat (obj, xl, xu, yl, yu, n)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRsurfgls SPKRexpcov SPKRgaucov SPKRsphercov 
;              SPKRprmat SPKRsemat SPKRcorrelogram SPKRvariogram
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRtrmat
; -----------------------------------------------------------------------
; Description  Evaluates a trend surface over a grid.
; -----------------------------------------------------------------------
; Usage        mygrid = SPKRtrmat (obj, xl, xu, yl, yu, n)
; Input
;   Parameter  obj
;   Definition list  -   (of type "trls" or "trgls") calculated by 
;                       SPKRsurfls or SPKRsurfgls
;   Parameter  xl
;   Definition scalar -  lower limit in x-direction of the rectangle for grid
;   Parameter  xu
;   Definition scalar  - upper limit in x-direction of the rectangle for grid
;   Parameter  yl
;   Definition scalar -  lower limit in y-direction of the rectangle for grid
;   Parameter  yu
;   Definition scalar -  upper limit in y-direction of the rectangle for grid
;   Parameter  n
;   Definition scalar -  use n x n grid within the rectangle
; Output
;   Parameter  res
;   Definition list  -   consisting of components 
;                       z, xl, xu, yl, yu, n, type --
;   Parameter  z
;   Definition grid  -   for further useage with SPKRmultcontours
;   Parameter  xl
;   Definition scalar -  same as input value xl
;   Parameter  xu
;   Definition scalar -  same as input value xu
;   Parameter  yl
;   Definition scalar -  same as input value yl
;   Parameter  yu
;   Definition scalar -  same as input value yu
;   Parameter  n
;   Definition scalar -  same as input value n
;   Parameter  type
;   Definition string -  "trmat"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       topo = read("topo.dat")
;	       ; calculate a polynomial regression surface of order 2
;  	       myres = SPKRsurfls (2, topo)
;  	       ; calculate a 30 x 30 grid within [0, 6.5] x [0, 6.5]
;  	       ; using trend surface myres
;  	       mygrid = SPKRtrmat (myres, 0, 6.5, 0, 6.5, 30)
; -----------------------------------------------------------------------
; Result       A grid suitable for display through SPKRmultcontours.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("obj") != 9, "obj must be a list")
  error (comp(obj, "type") == 0, "obj must contain component type")
  error ((obj.type != "trls") && (obj.type != "trgls"), "obj.type must be trls or trgls")
  error (exist("xl") != 1, "xl must be numeric")
  error (exist("xu") != 1, "xu must be numeric")
  error (exist("yl") != 1, "yl must be numeric")
  error (exist("yu") != 1, "yu must be numeric")
  error (exist("n") != 1, "n must be numeric")
  error (n < 1, "n must be >= 1")
  ;
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall("_VR_frset", obj.minx[1,1], obj.maxx[1,1], obj.minx[1,2], obj.maxx[1,2])
  else  
    ; Unix
    d = dlcall("VR_frset", obj.minx[1,1], obj.maxx[1,1], obj.minx[1,2], obj.maxx[1,2])
  endif
  ;
  dx = (xu - xl) / n
  dy = (yu - yl) / n
  x = grid (xl, dx, n+1)
  y = grid (yl, dy, n+1)
  z = grid (#(xl,yl, 0), #(dx, dy, 1), #(n + 1, n + 1, 1))
  i = 1
  do
    yrep = grid (y[i], 0, rows(x))
    ntrval = rows(x)
    ztrval = 1:ntrval
    ;
    if (os == "windows")
      d = dlcall("_VR_valn", ztrval, x, yrep, ntrval, obj.beta, obj.np)
    else  
      ; Unix
      d = dlcall("VR_valn", ztrval, x, yrep, ntrval, obj.beta, obj.np)
    endif
    ;
    z[(i-1)*rows(x)+1:i*rows(x), 3] = ztrval
    i = i+1
  until (i > rows(y))
  type = "trmat"
  res = list(z, xl, xu, yl, yu, n, type)
endp
