proc (res) = SPKRvariogram (krig, nint)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRsurfgls SPKRexpcov SPKRgaucov SPKRsphercov 
;              SPKRtrmat SPKRprmat SPKRsemat SPKRcorrelogram 
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRvariogram
; -----------------------------------------------------------------------
; Description  Computes spatial (semi-)variograms of spatial data or residuals.
;              Initially, it divides the range of the data into nint bins,
;              and computes the average squared difference for pairs with  
;              separation in each bin.
;              It returns results only for bins with 6 or more pairs.
; -----------------------------------------------------------------------
; Usage        varres = SPKRcorrelogram (krig, nint)
; Input
;   Parameter  krig
;   Definition list  -   (of type "trls" or "trgls") calculated by 
;                       SPKRsurfls or SPKRsurfgls
;   Parameter  nint
;   Definition scalar -  number of bins to be used
; Output
;   Parameter  res
;   Definition list  -   consisting of components 
;                       xpres, ypres, cntres --
;   Parameter  xpres
;   Definition n x 1  -  vector of the x coordinates of the variogram
;   Parameter  ypres
;   Definition n x 1  -  vector of the y coordinates of the variogram
;   Parameter  cntres
;   Definition n x 1  -  vector of the number of pairs averaged per bin
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       topo = read("topo.dat")
;	       ; calculate a polynomial regression surface of order 2
;  	       myres = SPKRsurfls (2, topo)
;  	       ; calculate the correlogram, using 25 bins
;  	       varres = SPKRvariogram (myres, 25)
; -----------------------------------------------------------------------
; Result       A list consisting of x and y coordinates of the variogram
;              and the number of pairs averaged per bin.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("krig") != 9, "krig must be a list")
  error (comp(krig, "type") == 0, "krig must contain component type")
  error ((krig.type != "trls") && (krig.type != "trgls"), "krig.type must be trls or trgls")
  error (exist("nint") != 1, "nint must be numeric")
  error (nint < 1, "nint must be >= 1")
  ;
  ;
  xp = 1:nint
  yp = 1:nint
  if (krig.np > 0)
    wzorz = krig.wz
  else
    wzorz = krig.z
  endif
  lengthx = rows(krig.x)
  cnt = 1:nint
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall("_VR_variogram", xp, yp, nint, krig.x, krig.y, wzorz, lengthx, cnt)
  else  
    ; Unix
    d = dlcall("VR_variogram", xp, yp, nint, krig.x, krig.y, wzorz, lengthx, cnt)
  endif
  ;
  xpres = xp[1:nint]
  ypres = yp[1:nint]
  cntres = cnt[1:nint]
  if (xpres[1] > 0)
    xpres = 0 | xpres
    ypres = 0 | ypres
    cntres = 0 | cntres
  endif
  res = list(xpres, ypres, cntres)
endp
