proc (res) = SPPPssi (nsim, n, r)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPPPinit SPPPinitrandom SPPPsetregion SPPPgetregion
;              SPPPkfn SPPPpsim SPPPstrauss SPPPkenvl SPPPkaver
; -----------------------------------------------------------------------
; Macro        SPPPssi
; -----------------------------------------------------------------------
; Description  Simulates a SSI (sequential spatial inhibition) point process.
;              Note that SPPPinit or SPPPsetregion must have been
;              called before to set the domain. To be able to reproduce 
;              results, reset the random number generator for point
;              processes by calling SPPPinitrandom first.
;              Note that this quantlet will never return a result
;              if r is too large and it is impossible to place n points.
; -----------------------------------------------------------------------
; Usage        ppssi = SPPPssi (nsim, n, r)
; Input
;   Parameter  nsim
;   Definition scalar -  number of simulations
;   Parameter  n
;   Definition scalar -  number of points in the domain
;   Parameter  r
;   Definition scalar -  inhibition distance
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       xmat, ymat, type --
;   Parameter  xmat
;   Definition n x nsim - matrix consisting of x coordinates 
;   Parameter  ymat
;   Definition n x nsim - matrix consisting of y coordinates 
;   Parameter  type
;   Definition string - "SSI"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       pines = read ("pines.dat")
;	       ; initialize a point process
;  	       pinesobj = SPPPinit (pines, 0, 96, 0, 100, 10)
;  	       ; reset random number generator
;              SPPPinitrandom (0)
;              ; simulate 100 SSI point processes (with r = 0.7),
;              ; each consisting of 72 points, in the previously
;              ; determined domain
;  	       ppssi = SPPPssi (100, 72, 0.7)
; -----------------------------------------------------------------------
; Result       List, consisting of x and y coordinates of 100 Strauss
;              point processes, each consisting of 72 points.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("nsim") != 1, "nsim must be numeric")
  error (nsim < 1, "nsim must be >= 1")
  error (exist("n") != 1, "n must be numeric")
  error (n < 1, "n must be >= 1")
  error (exist("r") != 1, "r must be numeric")
  error (r < 0, "r must be >= 0")
  ;
  ;
  area = SPPPgetregion ()
  xmat = matrix (n, nsim)
  ymat = matrix (n, nsim)
  x = 1:n
  y = 1:n
  ;
  i = 1
  os = getenv ("os")
  while (i <= nsim)
    ;
    if (os == "windows")
      d = dlcall ("_VR_simmat", n, x, y, r)
    else  
      ; Unix
      d = dlcall ("VR_simmat", n, x, y, r)
    endif
    ;
    xmat[,i] = x
    ymat[,i] = y
    i = i + 1
  endo
  type = "SSI"
  res = list (xmat, ymat, type)
endp
