proc (res) = SPPPstrauss (nsim, n, c, r)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPPPinit SPPPinitrandom SPPPsetregion SPPPgetregion
;              SPPPkfn SPPPpsim SPPPssi SPPPkenvl SPPPkaver
; -----------------------------------------------------------------------
; Macro        SPPPstrauss
; -----------------------------------------------------------------------
; Description  Simulates a Strauss spatial point process.
;              It uses a spatial birth-and-death process for (4 n) steps
;              (or for (40 n) steps when starting from a binomial pattern 
;              on the first call from another function).
;              Note that SPPPinit or SPPPsetregion must have been
;              called before to set the domain. To be able to reproduce 
;              results, reset the random number generator for point
;              processes by calling SPPPinitrandom first.
; -----------------------------------------------------------------------
; Usage        ppstrauss = SPPPstrauss (nsim, n, c, r)
; Input
;   Parameter  nsim
;   Definition scalar -  number of simulations
;   Parameter  n
;   Definition scalar -  number of points in the domain
;   Parameter  c
;   Definition scalar -  in [0, 1]; c = 0 corresponds to complete  
;                       inhibition at distances up to r
;   Parameter  r
;   Definition scalar -  inhibition distance
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       xmat, ymat, type
;   Parameter  xmat
;   Definition n x nsim - matrix consisting of x coordinates 
;   Parameter  ymat
;   Definition n x nsim - matrix consisting of y coordinates 
;   Parameter  type
;   Definition string - "STRAUSS"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       pines = read ("pines.dat")
;	       ; initialize a point process
;  	       pinesobj = SPPPinit (pines, 0, 96, 0, 100, 10)
;  	       ; reset random number generator
;              SPPPinitrandom (0)
;              ; simulate 100 Strauss point processes (with c = 0.15
;              ; and r = 0.7), each consisting of 72 points, 
;              ; in the previously determined domain
;  	       ppstrauss = SPPPstrauss (100, 72, 0.15, 0.7)
; -----------------------------------------------------------------------
; Result       List, consisting of x and y coordinates of 100 Strauss
;              point processes, each consisting of 72 points.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("nsim") != 1, "nsim must be numeric")
  error (nsim < 1, "nsim must be >= 1")
  error (exist("n") != 1, "n must be numeric")
  error (n < 1, "n must be >= 1")
  error (exist("c") != 1, "c must be numeric")
  error (c < 0, "c must be >= 0")
  error (exist("r") != 1, "r must be numeric")
  error (r < 0, "r must be >= 0")
  ;
  ;
  area = SPPPgetregion ()
  init = 1
  xmat = matrix (n, nsim)
  ymat = matrix (n, nsim)
  x = 1:n
  y = 1:n
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall ("_VR_pdata", n, x, y)
  else  
    ; Unix
    d = dlcall ("VR_pdata", n, x, y)
  endif
  ;
  i = 1
  while (i <= nsim)
    ;
    if (os == "windows")
      d = dlcall ("_VR_simpat", n, x, y, c, r, init)
    else  
      ; Unix
      d = dlcall ("VR_simpat", n, x, y, c, r, init)
    endif
    ;
    xmat[,i] = x
    ymat[,i] = y
    init = 0
    i = i + 1
  endo
  type = "STRAUSS"
  res = list (xmat, ymat, type)
endp
