proc(delta,dvar)=adeslp(x,y,d,m)
; -----------------------------------------------------------------
; Library        metrics
; -----------------------------------------------------------------
;  See_also      adeind wtsder trimper dwade
; -----------------------------------------------------------------
;   Macro        adeslp
; -----------------------------------------------------------------
;   Keywords     single index model, semiparametric estimation, 
;		 average derivative
; -----------------------------------------------------------------
;   Description   slope estimation of average derivatives 
;                 using binning           
; -----------------------------------------------------------------
;   Reference    Stoker (1991), "Equivalence of direct, indirect, and 
;		 slope estimators of average derivatives", 
;		 in: Nonparametric and semiparametric methods in 
;		 econometrics and statistics; 
;		 Barnett, Powell and Tauchen (eds.),Cambridge University Press
; -----------------------------------------------------------------
;   Usage        {delta,dvar} = adeslp(x,y,d,m)    
;   Input
;     Parameter   x  
;     Definition      n x p matrix , the observed explanatory variable    
;     Parameter   y 
;     Definition      n x 1 matrix , the observed response variable  
;     Parameter   d   
;     Definition      p x 1 vector or scalar , the binwidth or the grid
;     Parameter   m
;     Definition      p x 1 vector or scalar , the bandwidth to be used
;                                    during estimation of the scores 
;   Output                                                           
;     Parameter   delta 
;     Definition      p x 1 vector , the ADE estimate
;     Parameter   dvar
;     Definition      p x p matrix , the estimated asymptotic 
;                                    covariance matrix of delta  
; -----------------------------------------------------------------
;   Example   library("metrics")
;             randomize(0)
;             n   = 100
;             x   = normal(n,3)
;             z   = 0.2*x[,1] - 0.7*x[,2] + x[,3]
;             eps = normal(n,1) * sqrt(0.5)
;             y   = 2 * z^3 + eps
;             d   = 0.2
;             m   = 5
;             {delta,dvar} = adeslp(x,y,d,m)              
;             delta
;             dvar
; -----------------------------------------------------------------
;   Result    the slope estimator for average derivatives and its 
;             asymptotic covariance matrix  as described by
;             Stoker in Barnett, Powell, Tauchen, "Nonparametric and
;             Semiparametric Methods in Econometrics and Statistics"
;             (1991)          and Turlach, Discussion Paper (1993)
; -----------------------------------------------------------------
;   Author    Sperlich  960808  
; -----------------------------------------------------------------
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error(cols(y)!=1 ,"y has more than one column")
  p=cols(x)
  n = rows(x)
  if ( rows(m) .< p ) 
    m = matrix(p)*m
  endif 
  if ( rows(d) .< p ) 
    d = matrix(p)*d 
  endif
  wx=matrix(p)*0 
  {xb,yb} = bindata(x,d,wx,y)
  sel=(yb[,1].>0)
  xb=paf(xb,sel)
  yb=paf(yb,sel)
  vect= grid(1,1,cols(xb~yb)) 
  xb=sort(xb~yb,vect)
  fb=xb[,p+1]
  rb=xb[,p+2]
  xb=xb[,1:p]
  wts=symweigh(wx,(1./m),(m.+1),"qua")
  wx=grid(wx,matrix(p),(m.+1))
  {xc,yc,or}=conv(xb,fb,wx,wts)
  vect = grid(1,1,cols(xc~yc~or))
  xc2 = sort(xc~yc~or,vect)
   xc = xc2[,1:p]   
   yc = xc2[,p+1]
   or = xc2[,p+2]
  fh = paf(yc,or)
  xt = trimper(fh~fb~rb~xb,5)
  fh=xt[,1]
  fb=xt[,2]
  rb=xt[,3]
  xb=xt[,4:(p+3)]
  n=sum(fb) 
  fhder=matrix(rows(xb),p)*0
  i    = 1
  sym  = 1
  WHILE (i<= p)
    wts  = wtsder(p,m,i)
    {xc,yc,or} = conv(xb,fb,wx,wts,sym)
   xc2 = sort(xc~yc~or,vect) 
   xc = xc2[,1:p]   
   yc = xc2[,p+1]
   or = xc2[,p+2]
    fht  = paf(yc,or)
    fht  = fht / d[i,1]
    fhder[,i] = fht
    sym  = sym*2
    i    = i+1
  ENDO
  fhder = -fhder ./fh
  xt    = (xb.*d').*fb
  lhbar = (sum(fhder.*fb,1)/n)'
  xbar  = sum(xt,1)/n
xbar=xbar' 
  ybar  = sum(rb,1)/n
  slxi  = fhder'*xt/n - lhbar.*xbar'
  slxi  = inv(slxi)
  sly   = fhder'*rb/n - lhbar.*ybar'
  delta = slxi*sly
;
; Calculate variance
;
  u     = (rb-fb.*ybar') - (xt-fb.*xbar')*delta
  r     = matrix(rows(xb),p).*0
  i     = 1
  sym   = 1
  WHILE (i<= p)
    wts  = wtsder(p,m,i)
    {xc,yc,or} = conv(xb,u,wx,wts,sym)
    xc2 = sort(xc~yc~or,vect) 
    xc = xc2[,1:p]   
    yc = xc2[,p+1]
    or = xc2[,p+2]
    fht  = paf(yc,or)
    fht  = fht / d[i,1]
    r[,i] = fht
    sym  = sym*2
    i    = i+1
  ENDO
  r    = r./fh + (u./fb).*fhder 
  xbar = sum(r.*fb,1)/n
  r = r'*(r.*fb)/n - xbar.*xbar'
  dvar = slxi*r*slxi'
endp