proc(hcrit,crit)=denbwsel(x,h,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denest denci dencb denbwcrit denrot canker
; -----------------------------------------------------------------
;   Macro        denbwsel
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation, 
;                bandwidth selection
; -----------------------------------------------------------------
;   Description  interactive tool for bandwidth selection in  
;                univariate kernel density estimation.
; -----------------------------------------------------------------
;   Reference    Park and Turlach (1992): Practical performance of
;                several data driven bandwidth selectors
; 
;                Haerdle (1991): Smoothing Techniques 
; 
;                Haerdle, Mueller, Sperlich, Werwatz (1999): Non- and
;                Semiparametric Modelling
; -----------------------------------------------------------------
;   Usage        {hcrit,crit}= denbwsel(x {,h {,K} {,d} })
;   Input
;     Parameter  x  
;     Definition     n x 1 vector, the data.  
;     Parameter  h  
;     Definition     m x 1 vector, vector of bandwidths.
;     Parameter  K  
;     Definition     string, kernel function on [-1,1] or Gaussian
;                    kernel "gau". If not given, "gau" is used.
;     Parameter  d  
;     Definition     scalar, discretization binwidth. d must be smaller
;                    than h. If not given, the minimum of min(h)/3 and 
;                    (max(x)-min(x))/200 is used.
;   Output
;     Parameter  hcrit  
;     Definition     p x 1 vector, selected bandwidths by the
;                    different criteria.
;     Parameter  crit  
;     Definition     p x 1 string vector, criteria considered for bandwidth
;                    selection.
; -----------------------------------------------------------------
;   Notes     Internally, the Gaussian kernel is used for all
;             computations. If a different kernel function K is
;             given as input, the asymptotical equivalent
;             bandwidth for K is calculated via the canonical 
;             kernel approach.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=normal(500)
;             tmp=denbwsel(x)
;-----------------------------------------------------------------
;   Result    You may interactively choose the bandwidth
;             selector. The parameters (range of bandwidths,
;             kernel K, binwidth d) can be changed as well.
;-----------------------------------------------------------------
;   Author     Marco Bianchi & Sigbert Klinke, 930722 
;              Lijian Yang, 961026; Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  error (cols(x)<>1,"denbwsel: cols(x) <> 1")
  error (rows(x).<5,"denbwsel: at least 5 observations are required")    
  m=10      ; number of grid points  
  blanks="                              "
;
  if (exist(h)==0)
    silvh=denrot(x,"gau","robust") 
    hupp=1.05*silvh 
    hlow=silvh./1.05
    rangeofh=hlow|hupp
  else
    rangeofh = min(h)|max(h)
  endif
;
  if (exist(K)==0)
    K="gau"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="gau"
  endif
;
  if (exist(d)==0)
    d=(max(x)-min(x))./200
    d=min(d~(min(hlow)./3),2)
  endif
; 
  while (hupp<d)
    hupp = 2*hupp
  endo
  while (hlow<d)
    hlow = 2*hlow
  endo
  if (hupp>rangeofh[2])
    "denbwsel: upper bound for h was set to "+string("%10.6g",hupp)
  endif
  if (hlow>rangeofh[1])
    "denbwsel: lower bound for h was set to "+string("%10.6g",hlow)
  endif
  rangeofh = hlow|hupp
; 
  step=(hupp./hlow)            ; the bandwidth grid  
  step=log(step)./(m-1) 
  h=grid(log(hlow),step,m) 
  h=exp(h)
;
  stop=0
  while (stop==0)
    chra=0
    optionlist =            "Least Squares CV"
    optionlist = optionlist|"Biased CV"
    optionlist = optionlist|"Smoothed CV"
    optionlist = optionlist|"Jones, Marron and Park CV"
    optionlist = optionlist|"Park and Marron plug-in"
    optionlist = optionlist|"Sheather and Jones plug-in"
    optionlist = optionlist|"Silverman's rule of thumb"
    optionlist = optionlist|"(Change parameters)"
    optionlist = optionlist|"(Stop)"
    options = selectitem("Choose from the following", optionlist) 
    while (sum(options).>1)
      options = selectitem("Choose ONE from the following", optionlist)
    endo
    chosen=sum(options.*(1:rows(options)))
;
    switch
      case (chosen==1) ; Least Squares CV
        {hopt,ch}=denbwcrit("lscv",x,h,K,d)
        cc=substr(optionlist[1]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|1),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==2) ; Biased CV
        {hopt,ch}=denbwcrit("bcv",x,h,K,d)
        cc=substr(optionlist[2]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|2),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==3) ; Smoothed CV
        {hopt,ch}=denbwcrit("scv",x,h,K,d)
        cc=substr(optionlist[3]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|3),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==4) ; Jones, Marron and Park CV
        {hopt,ch}=denbwcrit("jmp",x,h,K,d)
        cc=substr(optionlist[4]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|4),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==5) ; Park and Marron plug-in
        {hopt,ch}=denbwcrit("pm",x,h,K,d)
        count=0
        while (hopt<0)
          count=count+1
          if (count<5)
            hupp=hupp*2
            hlow=max(3*d|hlow/2)
            message="denbwsel: Park and Marron plug-in didn't converge"
            message=message|"upper bound for h was set to "+string("%10.6g",hupp)
            message=message|"lower bound for h was set to "+string("%10.6g",hlow)
            step=(hupp./hlow)            ; the bandwidth grid  
            step=log(step)./(m-1) 
            h=grid(log(hlow),step,m) 
            h=exp(h)
            rangeofh = min(h)|max(h)
            {hopt,ch}=denbwcrit("pm",x,h,K,d)
          else
            hopt=NaN"denbwsel: cannot find good range for h!"
          endif
        endo
        cc=substr(optionlist[5]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|5),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==6) ; Sheather and Jones plug-in
        {hopt,ch}=denbwcrit("sj",x,h,K,d)
        count=0
        while (hopt<0)
          count=count+1
          if (count<5)
            hupp=hupp*2
            hlow=max(3*d|hlow/2)
            message="denbwsel: Sheather and Jones plug-in didn't converge"
            message=message|"upper bound for h was set to "+string("%10.6g",hupp)
            message=message|"lower bound for h was set to "+string("%10.6g",hlow)
            step=(hupp./hlow)            ; the bandwidth grid  
            step=log(step)./(m-1) 
            h=grid(log(hlow),step,m) 
            h=exp(h)
            rangeofh = min(h)|max(h)
            {hopt,ch}=denbwcrit("sj",x,h,K,d)
          else
            hopt=NaN
            "denbwsel: cannot find good range for h!"
          endif
        endo
        cc=substr(optionlist[6]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|6),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case  (chosen==7) ; Silverman's rule of thumb
        {hopt}=denrot(x,K)
        cc=substr(optionlist[7]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:7]+blanks,1,28)==crit').*(1:7),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|7),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case  (chosen==8) ; Change h
        rangetitle="Lower limit for h"|"Upper limit for h"
        rangetitle=rangetitle|"Grid points for h"|"Kernel K"|"Binwidth d"
        rangevalue=string("%-1.8g",rangeofh|m)|K|string("%-1.8g",d)
        notok=1
        while (notok)
          tmp=readvalue(rangetitle,rangevalue)
          r=atof(tmp[1:3|5])
          notok=(r[1]<=0)||(r[1]>=r[2])||(exist(tmp[4])<0)
          notok=notok||(r[4]>=r[1])||(r[4]>=r[1])
          if (r[1]<=0)
            "denbwsel: lower limit for h must be positive!"
          endif
          if (r[1]>=r[2])
            "denbwsel: lower limit for h must be smaller than upper limit!"
          endif
          if (exist(tmp[4])<0)
            "denbwsel: kernel function "+tmp[4]+" does not exist"
          endif
          if (r[4]>=r[1])
            "denbwsel: d must be smaller than lower limit for h!"
          endif
          if (r[4]<=0)
            "denbwsel: d must be positive!"
          endif
        endo
        rangeofh=r[1:2]
        m=r[3]
        K=tmp[4]
        d=r[4]
        hupp=r[2]
        hlow=r[1]
        step=(hupp./hlow)            ; the bandwidth grid  
        step=log(step)./(m-1) 
        h=grid(log(hlow),step,m) 
        h=exp(h)
        chra=1
        break
      default;
        stop=1
        break
    endsw
;
    hNum=1 ; hopt might not be defined ...
    if (exist(hopt)==1)
      if (hopt==NaN)
        hNum=0
      endif
    endif
;
    if (stop==0 && chra==0 && hNum)
      ComputedCriteria =  substr(crit+blanks,1,28)+string(" ... hopt = %10.6g",hcrit)
      ComputedCriteria 
      DensityBandwidthSelection=createdisplay(2,2)
      setfractions(DensityBandwidthSelection, 1|2, 7|3)
;
      range=" "|" Range of h:"|string("  %-10.6g", rangeofh)
      range=range|" "|" Points:"|string("  %-10.6g", m)
      range=range|" "|(" ---------- ")
      range=range|" "|" Kernel K:"|("  "+K)
d
      range=range|" "|" Binwidth d:"|string("  %-10.6g", d)
hopt
      opt=" "|" optimal h:"|""|string("  %-10.6g", hopt)
      show (DensityBandwidthSelection,1,2,opt)
;
      zeroline=rangeofh~(0|0)
      setmaskp(zeroline,0,0,0)
      setmaskl(zeroline,(1:rows(zeroline))',0,2,1)
;      
      switch
        case (chosen<5) ; minimize criterion
          show (DensityBandwidthSelection,2,2,range)
          setmaskp(ch,0,0,0)
          setmaskl(ch, (1:rows(ch))', 1, 1, 1)
          show (DensityBandwidthSelection,1,1,ch) 
          setgopt(DensityBandwidthSelection,1,1,"title",optionlist[chosen],"xlabel","h","ylabel","criterion") 
          break
        case (chosen==5 || chosen==6)
          show (DensityBandwidthSelection,2,2,range)
          setmaskp(ch,0,0,0)
          setmaskl(ch, (1:rows(ch))', 4, 1, 1)
          show (DensityBandwidthSelection,1,1,zeroline,ch) 
          setgopt(DensityBandwidthSelection,1,1,"title",optionlist[chosen],"xlabel","h","ylabel","criterion") 
          break
        default; silvermans ROT
          show (DensityBandwidthSelection,2,2,range[(rows(range))-3:rows(range)])
          show (DensityBandwidthSelection,1,1,""|(" "+optionlist[chosen]))
          break
      endsw
;
      fh=denest(x, hopt, K, d)
      setmaskp(fh,0,0,0)
      setmaskl(fh, (1:rows(fh))', 0, 1, 1)
      show (DensityBandwidthSelection,2,1, fh)
      setgopt(DensityBandwidthSelection,2,1,"title","Density Estimate","xlabel","x","ylabel","fh") 
    endif
  endo ; stop
endp