proc(fh)=denest(x,h,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denxest denci dencb denrot denbwsel denestp 
; -----------------------------------------------------------------
;   Macro        denest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation
; -----------------------------------------------------------------
;   Description  estimates a univariate density by kernel density
;                estimation. The computation uses WARPing.
; -----------------------------------------------------------------
;   References   Haerdle (1991): Smoothing Techniques 
; -----------------------------------------------------------------
;   Usage        fh = denest(x {,h {,K} {,d} })
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.
;     Parameter  h  
;     Definition   scalar, bandwidth. If not given, the rule of
;                  thumb bandwidth computed by denrot is used 
;                  (Silverman's rule of thumb).
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and 
;                  (max(x)-min(x))/100 is used. 
;   Output
;     Parameter  fh  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the density estimate on 
;                  that grid.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             mu = 10
;             si = 5
;             x  = si*normal(200)+mu          ; generate data
;             ;                                  
;             fh = denest(x)                  ; estimate density
;             f  = sort(x~pdfn((x-mu)/si)/si) ; true density                   ;             ;                                  
;             fh = setmask(fh,"line","blue")
;             f  = setmask(f ,"line","black","thin")
;             plot(f,fh)                      ; graph functions
; -----------------------------------------------------------------
;   Result    The density estimate (blue) for a normal distribution 
;             with mean mu=10, standard deviation si=5 is pictured 
;             using Quartic kernel (default) and Silverman's 
;             rule-of-thumb bandwidth (default), together with 
;             the true density (thin black).
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             mu = 10
;             si = 5
;             x  = si*normal(200)+mu           ; generate data   
;             ;                                  
;             fhe= denest(x,3,"epa")           ; estimate density
;             fhu= denest(x,3,"uni")           ; estimate density
;             f  = sort(x~pdfn((x-mu)/si)/si)  ; true density    
;             ;                              
;             fhe= setmask(fhe,"line","green")
;             fhu= setmask(fhu,"line","red")
;             f  = setmask(f ,"line","black","thin")
;             plot(f,fhu,fhe)                  ; graph functions
; -----------------------------------------------------------------
;   Result    The density estimate using the Epanechnikov kernel 
;             (green) is compared to the density estimate using
;             the Uniform kernel (red) and the true density (thin 
;             black). In both cases, bandwidth h=3 is used.
; -----------------------------------------------------------------
;   Author    Wolfgang Haerdle, 910426; Sigbert Klinke, 930219;
;             Lijian Yang, 980108; Marlene Mueller, 990413
; -----------------------------------------------------------------
  error(cols(x)<>1,"denest: cols(x) <> 1")
;
  n=rows(x)
  if (exist(h)==0)
    h=denrot(x); h=2.78*sqrt(var(x))*(n^(-0.2))
  endif
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
  if (exist(d)==0)
    d=(max(x)-min(x))/100
    d=min(d~(h./3),2)
  endif
;
  error(h .<=d,"denest: h smaller than d -- Please modify h or d!")
;
  {xb,yb}=bindata(x,d,0)            
  if (K!="gau")
    wy=symweigh(0,d/h,h/d,K)        
  else
    wy=symweigh(0,d/h,5.*h/d,K)        
  endif    
  wx=aseq(0,rows(wy),1)
  {xc,yc,or}=conv(xb,yb,wx,wy)
  fh=(d.*xc+d.*0.5)~(yc/(n*d))
endp


