proc(fh, clo, cup) = denxci(x,h,alpha,K,xv) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denxest denxcb denest denci dencb  
; -----------------------------------------------------------------
;   Macro        denxci
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation, 
;                confidence intervals
; -----------------------------------------------------------------
;   Description  computes pointwise confidence intervals with 
;                prespecified confidence level for univariate 
;                density estimation. 
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                exploratory purposes, the macro "denci" is
;                recommended, which uses the faster WARPing method.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        {fh, clo, cup} = denxci(x {,h {,alpha {,K} {,xv}}})
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, the rule of
;                  thumb bandwidth computed by denrot is used 
;                  (Silverman's rule of thumb).
;     Parameter  alpha  
;     Definition   confidence level, If not given, 0.05 is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1]. If not given,
;                  the Quartic kernel "qua" is used.
;     Parameter  xv  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the density estimate. If not given, 
;                  x is used.
;   Output
;     Parameter  fh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the
;                  second column contains the density estimate on 
;                  on the values of the first column.
;     Parameter  clo  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the
;                  second column contains the lower confidence 
;                  bounds on the values of the first column.
;     Parameter  cup  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted xv, the
;                  second column contains the upper confidence 
;                  bounds on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")                                       
;             x = 5*normal(200)+10
;             {fh, clo, cup} = denxci(x,3)
;             ;
;             fh=setmask(fh,"line","blue")
;             clo=setmask(clo,"line","blue","thin","dashed")
;             cup=setmask(cup,"line","blue","thin","dashed")
;             plot(fh,clo,cup)
; -----------------------------------------------------------------
;   Result    Pointwise confidence intervals at confidence level 
;             alpha = 0.05 for a normal density from N(10,25)  
;             are pictured using Quartic kernel (default) and 
;             bandwidth h=3.
; -----------------------------------------------------------------
;   Author    Marlene Mueller 990413
; -----------------------------------------------------------------
;
  error(cols(x)<>1,"denxci: cols(x) <> 1")
; 
  n=rows(x)
  if (exist(h)==0)
    h=denrot(x); h=2.78*sqrt(var(x))*(n^(-0.2))
  endif
  if (exist(alpha)==0)
    alpha = 0.05   ; default confidence level is 0.05    
  endif   
;
  if (exist(K)==0)
    cK=5/7
    K="qua"
  endif
  if (exist(K)==1) ; no K but xv
    xv=K
    cK=5/7
    K="qua"
  endif
  if (exist(K)==2)
    {dK,cK,mK}=canbw(K)
  endif
;
  if (exist(xv)==0)
    xv=x
  endif
;
  fh = denxest(x,h,K,xv)
;
  calpha =qfn(1-alpha/2)
  mrg = calpha .* sqrt(cK.*fh[,2]./(n*h))
  clo = fh[,1]~(fh[,2] - mrg)
  cup = fh[,1]~(fh[,2] + mrg)
endp  


