proc(fh)=denxest(x,h,K,v)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denest denci dencb denrot denbwsel denestp 
; -----------------------------------------------------------------
;   Macro        denxest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation
; -----------------------------------------------------------------
;   Description  estimates a univariate density by kernel density
;                estimation. 
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                exploratory purposes, the macro "denest" is
;                recommended, which uses the faster WARPing method.
; -----------------------------------------------------------------
;   References   Fan and Gijbels (1995): Local Polynomial Fitting
;                Fan and Marron (1994): Binning for local polynomials
;                Haerdle (1991): Smoothing Techniques 
; -----------------------------------------------------------------
;   Usage        fh = denxest(x {,h {,K} {,v} })
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.
;     Parameter  h  
;     Definition   scalar, bandwidth. If not given, the rule of
;                  thumb bandwidth computed by denrot is used 
;                  (Silverman's rule of thumb).
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  v  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the density estimate. If not given, 
;                  x is used.
;   Output
;     Parameter  fh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted v, the
;                  second column contains the density estimate on 
;                  on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             mu = 10
;             si = 5
;             x  = si*normal(200)+mu          ; generate data
;             ;                                  
;             fh = denxest(x)                  ; estimate density
;             f  = sort(x~pdfn((x-mu)/si)/si) ; true density                   ;             ;                                  
;             fh = setmask(fh,"line","blue")
;             f  = setmask(f ,"line","black","thin")
;             plot(f,fh)                      ; graph functions
; -----------------------------------------------------------------
;   Result    The density estimate (blue) for a normal distribution 
;             with mean mu=10, standard deviation si=5 is pictured 
;             using Quartic kernel (default) and Silverman's 
;             rule-of-thumb bandwidth (default), together with 
;             the true density (thin black).
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             mu = 10
;             si = 5
;             x  = si*normal(200)+mu           ; generate data   
;             ;                                  
;             fhe= denxest(x,3,"epa")           ; estimate density
;             fhu= denxest(x,3,"uni")           ; estimate density
;             f  = sort(x~pdfn((x-mu)/si)/si)  ; true density    
;             ;                              
;             fhe= setmask(fhe,"line","green")
;             fhu= setmask(fhu,"line","red")
;             f  = setmask(f ,"line","black","thin")
;             plot(f,fhu,fhe)                  ; graph functions
; -----------------------------------------------------------------
;   Result    The density estimate using the Epanechnikov kernel 
;             (green) is compared to the density estimate using
;             the Uniform kernel (red) and the true density (thin 
;             black). In both cases, bandwidth h=3 is used.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; -----------------------------------------------------------------
  error(cols(x)<>1,"denxest: cols(x) <> 1")
;
  n=rows(x)
  if (exist(h)==0)
    h=denrot(x); h=2.78*sqrt(var(x))*(n^(-0.2))
  endif
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but v
    v=K
    K="qua"
  endif
;
  x=sort(x)
  if (exist(v)==0)
    v=x
  else
    v=sort(v)
  endif
;
  if (K!="gau")
    tmp=sker(x,h,K,matrix(n),v)
  else
    tmp=sker(x,5.*h,"ngau",matrix(n),v)
  endif
  fh=v~tmp
endp


