proc(mux,beta1,beta0,sigmax,sigmau,sigmae,varbeta1,varbeta0)=eivknownatt(w,y,kww)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also	 eivknownvaru eivknownratue
; -----------------------------------------------------------------
;    Macro	eivknownatt
; -----------------------------------------------------------------
;    Description    eivknownatt presents the moment estimates of the 
;                   parameters in the measurement error models, 
;                   which has only one variable x. The degree of 
;                   attenuation (also called reliability ratio) is 
;                   known. All of the variables obey normal 
;                   distributions.
; ----------------------------------------------------------------------------
;   Reference	      Fuller W. A. "Measurement Error Models", Wiley and Sons
;                   1987, section 1.1.
; -----------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in XploRe 
; -----------------------------------------------------------------
;    Usage    {mux,beta1,beta0,sigmax,sigmau,sigmae,varbeta1,varbeta0} = eivknownatt(w,y,kww)
;    Input
;      Parameter  w
;      Definition   n x 1 matrix, the design variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;      Parameter  kww
;      Definition   scalar, the degree of attenuation
;    Output
;      Parameter  mux
;      Definition   scalar, the mean value of x
;      Parameter  beta1
;      Definition   scalar, the estimate
;      Parameter beta0
;      Definition  scalar, the estimate
;      Parameter sigmax
;      Definition  scalar, the estimate of the variance of x
;      Parameter sigmau
;      Definition  scalar, the estimate of the variance of u
;      Parameter sigmae
;      Definition  scalar, the estimate of the variance of e
;      Parameter  varbeta1   
;      Definition  scalar, the estimate of the variance of the estimate of beta1
;      Parameter  varbeta0   
;      Definition  scalar, the estimate of the variance of the estimate of beta0         
; -----------------------------------------------------------------
;  Example library("eiv")
;          n = 100
;          randomize(n) 
;          x=0.9*normal(n)
;          w=x+0.9*normal(n)
;          y=0.9+0.8*x+0.01*normal(n)
;          kww =0.5
;          gest=eivknownatt(w,y,kww)
;          gest.mux        ; the estimate of the mean of x
;          gest.beta1      ; the estimate of b (true value is 0.8)
;          gest.beta0      ; the estimate of a (true value is 0.9)
;          gest.sigmax     ; the estimate of the variance of x
;          gest.sigmau     ; the estimate of the variance of u
;          gest.sigmae     ; the estimate of the variance of e
;          gest.varbeta1   ; the estimate of the variance of the estimate of beta1
;          gest.varbeta0   ; the estimate of the variance of the estimate of beta0
; -----------------------------------------------------------------
;   Result Contents of mux
;          [1,] -0.93396 
;          Contents of beta1
;          [1,]   0.79286
;          Contents of beta0
;          [1,]   0.8425 
;          Contents of sigmax
;          [1,]   0.72585
;          Contents of sigmau
;          [1,]   0.72585
;          Contents of sigmae
;          [1,]   0.074451
;          Contents of varbeta1
;          [1,]   0.0085078
;          Contents of varbeta0
;          [1,]   0.0054358
; -----------------------------------------------------------------
;   Author    Hua Liang,  990625
; -----------------------------------------------------------------
     n=rows(w)
     mux=mean(w)
     mww=var(w)
     mwy=sum((w-mean(w)).*(y-mean(y)))/(rows(w)-1)
     myy=var(y)
     gamma=inv(mww)*mwy
     beta1=inv(kww)*gamma
     beta0=mean(y)-beta1*mean(w)
     sigmax=mwy*inv(beta1)
     sigmau=mww-sigmax
     sigmae=myy-kww*beta1^2*mww
     sl=inv(n-2)*sum((y-mean(y)-gamma*(w-mean(w))).^2)
     svv=inv(n-2)*sum((y-beta0-beta1*w).^2)
     varbeta1=kww^(-2)*sl*inv(sum((w-mean(w)).^2))
     varbeta0=inv(n)*svv+(mean(w))^2*varbeta1
endp
