proc(hatbeta,sigmam)=eivvec2(w,y,Gamma)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also   eivvec1
; -----------------------------------------------------------------
;    Macro	eivvec2
; -----------------------------------------------------------------
;    Description    eivvect2 calculates the maximum likelihood estimators of
;		    the parameters in the measurement error models when
;		    the entire error covariance structure is known
;		    or known up to a scalar multiple. This macro deals
;		    with the extension of the model considered in
;		    eivknownvaru.
; ----------------------------------------------------------------------------
;   Reference	 Fuller W. A. "Measurement Error Models", Wiley and Sons 1987,
;              page 124-126. 
; -----------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in wploRe 
; -----------------------------------------------------------------
;    Usage    {hatbeta,sigmam)=eivvec2(w,y,Gamma)
;    Input
;      Parameter  w
;      Definition   n x p matrix, the design variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;      Parameter  Gamma
;      Definition   (p+1) x (p+1) matrix
;    Output
;      Parameter  hatbeta
;      Definition   vector, the estimate
;      Parameter sigmam
;      Definition  scalar, the estimate
; -----------------------------------------------------------------
;  Example library("xplore")
;	   library("eiv")
;	   y=#(1.015, 1.120, 1.937, 1.743, 2.173)
;	   w=0*matrix(5,3)
;	   w[,1]=#(1,1,1,1)
;	   w[,2]=#(3.442, 3.180,3.943, 3.983, 4.068)
;	   w[,3]=#(0.477, 0.610, 0.505, 0.415, 0.620)
;	   sig=0*matrix(4,4)
;	   sig[,1]=#(0.5546, 0,-0.1079, -0.0691)
;	   sig[,2]=#(0,0,0,0)
;	   sig[,3]=#(-0.1079,0,0.2756,0.1247)
;	   sig[,4]=#(-0.0691,0,0.1247,0.0878)
;	   Gamma=0.01*sig
;	   gest=eivvec2(w,y,Gamma)
;	   gest.hatbeta
;	   gest.sigmam
; -----------------------------------------------------------------
;    Result  Contents of hatbeta
;            [1,] -2.7727e+13
;            [2,]  0.24253
;            [3,]        0 
;            Contents of sigmam
;            [1,]  0.030575 
; -----------------------------------------------------------------
;   Author    Hua Liang,  970724
; --------------------------------------------------------
library("xplore")
     mux=mean(w)
     mww=cov(w)
     mwy=mean((w-mean(w)).*(y-mean(y)))/(rows(w)-1)
     myy=var(y)
     z=y~w
     n=rows(w)
     mzz=(z'*z)/n
     p=cols(w)
   if(det(Gamma)==0)
     {eval,evec}=eigsm(Gamma*inv(mzz))
     eval=sort(eval,1)
     hatlam=inv(eval[1+p])
   else
     {eval,evec}=eigsm(mzz*inv(Gamma))
     eval=sort(eval,1)
     hatlam=eval[1]
    endif
     Gammau=Gamma[2:p+1,2:p+1]
     Gammaue=Gamma[2:p+1,1]
     hatbeta=inv(mww-hatlam*Gammau)*(mwy'-hatlam*Gammaue)
     sigmam=hatlam/(p+1)
endp
