proc(xs,ts,ys,wx,wt,wc,b0,m0,off,ctrl,tg,m0g,tro,tgro)=gplminit(code,x,t,y,h,opt) 
; ----------------------------------------------------------------------------
; Library       gplm
; ----------------------------------------------------------------------------
;  See_also     gplmopt gplmcore gplmest 
; ----------------------------------------------------------------------------
;   Macro       gplminit
; ----------------------------------------------------------------------------
;   Description  gplminit checks the validity of input and performs the 
;                initial calculations for an GPLM fit (inclusive sorting).
;                The output is ready to be used with gplmcore.
; ----------------------------------------------------------------------------
;   Link   ../tutorials/gplmstart.html Tutorial: GPLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        init = gplminit(code,x,t,y,h{,opt})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the discrete predictor variables.
;     Parameter   t  
;     Definition        n x q  matrix, the continuous predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.  
;     Parameter   h  
;     Definition        q x 1  vector, the bandwith.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "gplmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.b0  
;     Definition        p x 1  vector, the initial coefficients. If not 
;                       given, all coefficients are put =0 initially.
;     Parameter   opt.m0  
;     Definition        n x 1  vector, the initial values for the nonparametric
;                       part. If not given, a default is used.
;     Parameter   opt.tg  
;     Definition        ng x 1 vector, a grid for continuous part. If tg is
;                       given, the nonparametric function will also be 
;                       computed on this grid.
;     Parameter   opt.m0g  
;     Definition        ng x 1  vector, the initial values for the 
;                       nonparametric part on the grid. These values are 
;                       ignored if direct update for nonparametric function 
;                       is possible. Otherwise, if not given, it is 
;                       approximated from m0.
;     Parameter   opt.weights
;     Definition        string, type of observation weights. Can be 
;                       "frequency" for replication counts, or "prior" 
;                       (default) for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.wc 
;     Definition        n x 1  vector, weights for convergence criterion,
;                       w.r.t. m(t) only. If not given, opt.wt is used. 
;     Parameter   opt.wt  
;     Definition        n x 1  vector, weights for t (trimming factors). 
;                       If not given, all set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset. Can be used for
;                       constrained estimation. If not given, set to 0.
;     Parameter   opt.meth
;     Definition        integer, if -1, a backfitting is performed,
;                       if 1 a profile likelihood method is used, and
;                       0 a simple profile likelihood is used. 
;                       The default is 0.
;     Parameter   opt.fscor
;     Definition        integer, if exists and =1, a Fisher scoring is
;                       performed (instead of the default Newton-Raphson
;                       procedure). This parameter is ignored for
;                       canonical links.
;     Parameter   opt.miter
;     Definition        integer, maximal number of iterations. The default 
;                       is 10.
;     Parameter   opt.cnv
;     Definition        integer, convergence criterion. The default is 0.0001.
;     Parameter   opt.shf  
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the iteration is going on.
;     Parameter   opt.nosort
;     Definition        integer, if exists and =1, the continuous variables
;                       t and the grid tg are assumed to be sorted by the 
;                       1st column. Sorting is required by the algorithm,
;                       hence you should switch if off only when the data 
;                       are already sorted.
;     Parameter   opt.pow
;     Definition        optional, power for power link.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   init.xs  
;     Definition        n x p  matrix, x sorted according to 1st column of t.
;     Parameter   init.ts  
;     Definition        n x q  matrix, t sorted according to its 1st column.
;     Parameter   init.ys  
;     Definition        n x 1  vector, x sorted according to 1st column of t.
;     Parameter   init.wx
;     Definition        n x 1 vector or scalar, prior weights.
;     Parameter   init.wt  
;     Definition        n x 1  vector or scalar, weights for t (trimming 
;                       factors). 
;     Parameter   init.wc 
;     Definition        n x 1  vector or scalar, weights for convergence 
;                       criterion.
;     Parameter   opt.wr  
;     Definition        n x 1  vector, weights for test statistics.
;                       If not given, set to 1.
;     Parameter   opt.tdesign  
;     Definition        n x r matrix, design for parametric fit for
;                       m(t) in gplmbootstratest. 
;     Parameter   init.b0  
;     Definition        p x 1 vector, the initial coefficients. 
;     Parameter   init.m0  
;     Definition        n x 1 vector or scalar, the initial values for the 
;                       nonparametric part. Set =NaN if direct update 
;                       for nonparametric function is possible (code="noid").
;     Parameter   init.off
;     Definition        n x 1 vector or scalar, offset.
;     Parameter   init.ctrl  
;     Definition        7 x 1 vector or scalar, contains control parameters
;                       shf, miter, cnv, fscor, pow (for the power link
;                       function), nbk (for negative binomial distribution)
;                       and meth (parameter for backfitting/profile).
;                       They correspond to the optional parameters which can 
;                       be given in gplmest.
;                       Defaults are 0|10|0.0001|0|0|1|0. 
;     Parameter   init.tg  
;     Definition        ng x 1 vector, a grid for continuous part. 
;                       Is =NaN, if opt.tg was empty or unusable.
;     Parameter   init.m0g  
;     Definition        ng x 1 vector or scalar, the initial values 
;                       for the nonparametric part on the grid. Is =NaN, 
;                       if opt.tg was empty.
;     Parameter   init.tro  
;     Definition        n x 1 vector, "re"order index for t. 
;                       Is =NaN, if opt.nosort was 1 or empty.
;     Parameter   init.tgro  
;     Definition        ng x 1 vector, "re"order index for tg. 
;                       Is =NaN, if opt.tg was empty or if opt.nosort was 
;                       1 or empty.
; ----------------------------------------------------------------------------
;   Example   library("gplm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             t=sort(2.*uniform(n)-1,1)
;             m=cos(pi.*t)
;             y=( 1./(1+exp(-x*b-m)).>uniform(n) )
;             ;==========================
;             ;  semiparametric fit 
;             ;==========================
;             h=0.6
;             opt=gplmopt("shf",1)
;             {xs,ts,ys,wx,wt,wc,b0,m0,off,ctrl,tg,m0g,ro,rgo}=gplminit("bilo",x,t,y,h,opt)
;             sf=gplmcore("bilo",xs,ts[,1],ys,h,wx,wt,wc,b0,m0,off,ctrl,1,tg,m0g)
;             ;==========================
;             ;  plot
;             ;==========================
;             library("plot")
;             true=setmask(t~m,"line","thin")
;             estm=setmask(t~sf.m,"line","blue")
;             plot(true,estm)
; ----------------------------------------------------------------------------
;   Result    A generalized partially linear logit fit for E[y|x,t] is 
;             computed. sf.b contains the coefficients for the linear  
;             part. sf.m contains the estimated nonparametric part 
;             evaluated at observations t. The example gives the 
;             true b together with the GPLM estimate sf.b. Also, the  
;             estimated function sf.m is displayed together with the 
;             true fit m. 
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/10
; ----------------------------------------------------------------------------
;
; classify our algo
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  direct    = sum(code==(glmmodels.direct)) >0
  power     = sum(code==(glmmodels.power)) >0
;
; check initial errors
;
  bad=(rows(dim(x))>2)+(rows(dim(t))>2)
  error(bad>0,"x and t must be vectors or matrices")
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error(rows(x)!=rows(t),"x and t have different number of rows")
  error((rows(dim(y))>1),"y must be a vector")
;
  n=rows(x)
  p=cols(x)
  q=cols(t)
  one=matrix(n)
  h=vec(h)
  bad=(rows(h)!=q)&&(rows(h)!=1)
  error(bad>0,"bandwidth h has wrong dimension")
  h=h.*matrix(q)
;
; set the defaults
;
  b0=NaN;matrix(p)-1
  wt=1
  wc=1
  wx=1
  wr=one
  off=0
  nosort=0
  havegrid=0
  shf=0
  miter=10
  cnv=0.0001
  fscor=0
  pow=0
  nbk=1
  meth=0
  weights="prior"
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"b0")>0)
      notgood=(rows(opt.b0)!=p)+(rows(dim(opt.b0))!=1)
      warning (notgood>0, "opt.b0 not consistent with x, used zeros instead")
      if (notgood==0)
        b0=opt.b0
      endif
    endif
    if (comp(opt,"m0")>0)
      notgood=(rows(opt.m0)!=n)+(rows(dim(opt.m0))!=1)
      warning (notgood>0, "opt.m0 not consistent with t, used default instead")
      if (notgood==0)
        m0=opt.m0
      endif
    endif
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(opt.wx))==0)+(min(opt.wx)<0)
      warning(notgood>0, "opt.wx was unusable, used opt.wx=1 instead")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      warning(notgood>0, "opt.off was unusable, used opt.off=0 instead")
      if (notgood==0)
        off=opt.off
      endif
    endif
    if (comp(opt,"wt")>0)
      notgood=(dim(dim(opt.wt))!=1)*(rows(opt.wt)!=n)+(rows(dim(opt.wt))!=1)
      warning(notgood>0, "opt.wt not consistent with t, used wt=1 instead")
      if (notgood==0)
        wt=opt.wt
      endif
    endif
    if (comp(opt,"wc")>0)
      notgood=(dim(dim(opt.wc))!=1)*(rows(opt.wc)!=n)+(rows(dim(opt.wc))!=1)
      warning(notgood>0, "opt.wc not consistent with t, used wc=1 instead")
      if (notgood==0)
        wc=opt.wc
      endif
    endif
    if (comp(opt,"wr")>0)
      notgood=(dim(dim(opt.wr))!=1)*(rows(opt.wr)!=n)+(rows(dim(opt.wr))!=1)
      warning(notgood>0, "opt.wr not consistent with t, used wr=1 instead")
      if (notgood==0)
        wr=opt.wr
      endif
    endif
    if (comp(opt,"tg")>0)
      notgood=(cols(opt.tg)!=q)+(rows(dim(opt.tg))>2)
      warning(notgood>0, "opt.tg not consistent with t, was not used")
      if (notgood==0)
        havegrid=1
        tg=opt.tg
        ng=rows(tg)
      endif
    endif
    if (comp(opt,"m0g")>0)
      warning(havegrid==0,"since opt.tg missing or bad, opt.m0g was not used")
      if (havegrid==1)
        notgood=(rows(opt.m0g)!=ng)+(rows(dim(opt.m0g))>2)
        warning(notgood>0, "opt.m0g not consistent with tg, was not used")
        if (notgood==0)
          m0g=opt.m0g
        endif
      endif
    endif
    if (comp(opt,"nosort")>0)
      nosort=(opt.nosort==1)*(dim(dim(opt.nosort))==1)
    endif
    if (comp(opt,"shf")>0)
      shf=(opt.shf==1)*(dim(dim(opt.shf))==1)
    endif
    if (comp(opt,"fscor")>0)
      fscor=(opt.fscor==1)*(dim(dim(opt.fscor))==1)
    endif
    if (comp(opt,"meth")>0)
      meth=(opt.meth)*(dim(dim(opt.meth))==1)
    endif
    if (comp(opt,"miter")>0)
      tmp=floor(opt.miter[1,1])
      notgood=(tmp<1)+(dim(dim(opt.miter))!=1)
      warning (notgood>0, "opt.miter was unusable, used default =10 instead")
      miter=notgood*miter+(1-notgood)*tmp
    endif
    if (comp(opt,"cnv")>0)
      tmp=opt.cnv[1,1]
      notgood=(tmp<=0)+(dim(dim(opt.cnv))!=1)
      warning (notgood>0, "opt.cnv was unusable, used default =0.0001 instead")
      cnv=notgood*cnv+(1-notgood)*tmp
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"tdesign")>0)
      notgood=(rows(opt.tdesign)!=n)+(rows(dim(opt.tdesign))>2)
      warning (notgood>0, "opt.tdesign not consistent with t, not used")
      if (notgood==0)
        tdesign=opt.tdesign
      endif
    endif
  endif
;
; check y with respect to model
;
  switch
    case (binomial)
      error(min(y) <0 ,"glm/code='"+code+"': y has negative values")
      error(max(y)==0 ,"glm/code='"+code+"': y has wrong values")
      if (weights=="prior")
        if (max(paf(y,wx.*one))>1)
          error(sum(y>wx)!=0,"glm/code='"+code+"': y has wrong values")
          y[,1] = y[,1]./(wx+(wx==0)) ; divide by wx, but not by 0!
        endif
      endif
      break
    case (poisson)
      error(min(y)<0 ,"glm/code='"+code+"': y has negative values")
      break
    case (gamma)
      error(min(y)<=0,"glm/code='"+code+"': y has non-positive values")
      break
    case (igaussian)
      error(min(y)<=0,"glm/code='"+code+"': y has non-positive values")
      break
    case (nbinomial)
      error(min(y)<0 ,"glm/code='"+code+"': y has negative values")
      break
    default
      ; all is allowed ...
  endsw
;
; we (eventually) sort data by t, required by gplm.so and/or sker!
;
  if (havegrid!=1)
    tg=NaN
  endif
;
  if (nosort!=1)
    to=sort(t[,1]~(1:n),1)[,2]
    tro=sort(to~(1:n),1)[,2]
    xs=x[to]
    ys=y[to]
    ts=t[to]
    if (exist(m0))
      m0=m0[to]
    endif
    if (exist(tdesign)&&exist(wr))
      tdesign=tdesign[to]~wr[to]
    endif
    if (!exist(tdesign)&&exist(wr))
      tdesign=wr[to]
    endif
    if (rows(wt)==n)
      wt=wt[to]
    endif
    if (rows(wc)==n)
      wc=wc[to]
    endif
    if (rows(wx)==n)
      wx=wx[to]
    endif
    if (rows(off)==n)
      off=off[to]
    endif
    if (havegrid==1)
      tgo=sort(tg[,1]~(1:ng))[,2]
      tgro=sort(tgo~(1:ng))[,2]
      tg=tg[tgo]
      if (exist(m0g))
        m0g=m0g[tgo]
      endif
    else
      tgro=NaN
    endif
  else
    xs=x
    ys=y
    ts=t
    tro=NaN
    tgro=NaN
  endif
;
; check x and create initial values, if necessary
;
  error (countNaN(inv(xs'*(wx.*xs)))>0,"gplminit: Matrix x is singular!")
;
  if (rows(wx)==1)
    wx=wx.*one
  endif
;
;
; define control parameter vector
;
  ctrl   =0.*matrix(7)
  ctrl[1]=shf
  ctrl[2]=miter
  ctrl[3]=cnv
  ctrl[4]=fscor*(1-canonical)
  ctrl[5]=pow
  ctrl[6]=nbk
  ctrl[7]=meth
;
  if (exist(m0)==0)
    m0g=NaN
    if (countNaN(b0)==prod(dim(b0)))
      tmp=ts~one
      {b,bv,it,ret}=glmcore(code,xs~tmp,ys,wx,off,ctrl[1:6])
      b0=b[1:p]
      m0=ts*b[(p+1):(p+q)]+b[p+q+1]
      if ((havegrid)&&(direct!=1))
        m0g=tg*b[(p+1):(p+q)]+b[p+q+1]
      endif
    else
      off=x*b0
      tmp=ts~one
      {b,bv,it,ret}=glmcore(code,tmp,ys,wx,off,ctrl[1:6])
      m0=t*b[1:q]+b[q+1]
      if ((havegrid)&&(direct!=1))
        m0g=tg*b[1:q]+b[q+1]
      endif
    endif
  endif
;    switch
;      case (binomial)  
;        if (weights=="prior")
;          m0=glminvlink(code,(ys.*wx+0.5)./(wx+1),list(pow,nbk))-xs*b0
;        else
;          m0=glminvlink(code,(ys+0.5)./2,list(pow,nbk))-xs*b0
;	endif 
;	break
;      case (direct)    
;	m0=NaN                        
;	break
;      default          
;	m0=glminvlink(code,ys,list(pow,nbk))-xs*b0
;	break
;    endsw
;    m0g=NaN
;    if ((havegrid==1)*(direct!=1))
;    tmp=sker(ts,h',"qua",one~m0,tg)
;    m0g=tmp[,2]./tmp[,1]
;  endif
  if (exist(tdesign))
    ts=ts~tdesign
  endif
endp 
