proc(g, gp)=grcarttree(tr,string)
; -----------------------------------------------------------------------
; Library       xclust
; -----------------------------------------------------------------------
; See_also      cartsplit, dispcarttree, grcarttree, grcart2
; -----------------------------------------------------------------------
; Macro         grcarttree
; -----------------------------------------------------------------------
; Description   Plots the regression tree of CART in a display
; -----------------------------------------------------------------------
; Usage         {g, gp} = grcarttree(ts,stri)
; Input        
;   Parameter   cs
;   Definition  list of vectors: data structure which represents a binary tree
;               and is produced by cartsplit procedure, contains vectors ..
;               cs.val, cs.vec, cs.mean, cs.ssr, cs.nelem.
;               See cartsplit for the description of cs.
;   Parameter   stri
;   Definition  string: optional argument,
;               either missing, "nelem", "ssr" or "mean".
; Output
;   Parameter   g
;   Definition  CART tree
;   Parameter   gp
;   Definition  text labels for CART tree
; -----------------------------------------------------------------------
; Notes 
;               The content of the nodes in the picture of the tree 
;               is determined by the parameter stri.
;               If this parameter is not given, in the nonterminal 
;               nodes there
;               are given the splitting rule 
;               (for example x1<=1.34), and in the leaf nodes there are
;               given the values of the regression function, that is,
;               the mean of the observations in that node.
;               If stri = "nelem", 
;               the number of observations in the nodes is plotted,
;               if stri = "ssr",
;               the sum of squared residuals in the nodes is plotted,
;               and if stri = "mean",
;               the mean of observations in the nodes is plotted. 
; -----------------------------------------------------------------------
; Example      ; loads the library xclust
;              library ("xclust")
;              ; generate an artifcal dataset
;              x1=#(0,0,0,0,1,1,1,1,1,2)
;              x2=#(0,0,0,0,0,0,0,1,1,1)
;              x=x1~x2
;              y=#(0,0,0,0,100,100,100,120,120,120)
;              ; run CART on our data
;              cs=cartsplit(x,y,#(0,1))
;              ; plot CART tree
;              d = createdisplay(1,1)
;              {g, gp} = grcarttree (cs)
;              show (d, 1, 1, g, gp)
; -----------------------------------------------------------------------
; Result       shows the tree with the splitting rules
; ---------------------------------------------------------------------
; Keywords     CART
; ---------------------------------------------------------------------
; Reference    Breiman, L., Friedman, J., Olshen, R., and Stone, C. J. (1984). 
;              Classification and Regression Trees. Chapman & Hall, New York
; ---------------------------------------------------------------------
; Link         ..\tutorials\cartstart.html CART Tutorial
; ---------------------------------------------------------------------
; Author       1999 Jussi Klemelae, 2000-04-15, Sigbert Klinke
; ---------------------------------------------------------------------
  if (exist("string").=0)
    string = "empty"
  endif
; use for y-axis the variances  
  sv=(log10(1+tr.ssr)).*(tr.vec/tr.vec)
; build tree
  t=tree(sv, 0, "CENTER")
  g=t.points
  cond=(g[,2].=0)
  g[,2]=g[,2]+cond.*(min(sv)-0.1*(max(sv)-min(sv)))
; build lines 
  l = 5.*(1:rows(g)/5) + (0:4)' - 4 
  setmaskl (g, l, 0, 1, 1) 
  setmaskp (g, 0, 0, 0) 
; create some additional information
  gp = paf (g, t.index)
  gpi = paf (t.index, t.index)
  switch 
  case (string=="nelem")
    n=string ("%.0f", tr.nelem[gpi+1])
    n=n|string ("%.0f", tr.nelem[1])
    gp=gp|mean(paf(g, g[,2].=max(g[,2])))
    setmaskp (gp, 0, 0, 0) 
    setmaskt (gp, n, 0, 0, 12)
    break
  case (string=="ssr")
    n=string ("%.2f", tr.ssr[gpi+1])
    n=n|string ("%.0f", tr.ssr[1])
    gp=gp|mean(paf(g, g[,2].=max(g[,2])))
    setmaskp (gp, 0, 0, 0) 
    setmaskt (gp, n, 0, 0, 12)
    break
  case (string=="mean")
    n=string ("%.2f", tr.mean[gpi+1])
    n=n|string ("%.0f", tr.mean[1])
    gp=gp|mean(paf(g, g[,2].=max(g[,2])))
    setmaskp (gp, 0, 0, 0) 
    setmaskt (gp, n, 0, 0, 12)
    break
;case (string=="defa")
  default
    defa=muunna(tr,tr.mean)
    n=string ("%s", defa.left[gpi+1])|string ("%s", defa.right[gpi+1])
    n=n|defa.left[1]|defa.right[1]
    gp = gp|gp
    pt = g
    pt = mean(paf(pt, pt[,2].=max(pt[,2])))
    gp = gp|pt|pt
    setmaskp (gp, 0, 0, 0) 
    cond  = isNaN(tr.vec) || defa.cond
    label = 9-9.*cond[gpi+1]
    label = label|(3-3.*cond[gpi+1])
    if (rows(tr.val.split0)>1)
      label = label|9|3
    else
      label = label|0|0
    endif
    setmaskt (gp, n, 0, label, 12)
    break
  endsw 
endp
