proc(draw)=grface(obsrv, p, fl)
; -----------------------------------------------------------------------
; Library      graphic
; -----------------------------------------------------------------------
; Macro        grface
; -----------------------------------------------------------------------
; See_also     plotface
; -----------------------------------------------------------------------
; Keywords     multivariate analysis, Flury faces
; -----------------------------------------------------------------------
; Description  calculates Flury faces                     
; -----------------------------------------------------------------------
; Notes		List of face parameters (right, left): 
;       eye size                        (1,       19),
;       pupil size                      (2,       20),
;       position of pupil               (3,       21),
;       eye slant                       (4,       22),
;       horizontal position of eye      (5,       23),
;       vertical position of eye        (6,       24),
;       curvature of eyebrow            (7,       25),
;       density of eyebrow              (8,       26),
;       horizontal position of eyebrow  (9,       27),
;       vertical position of eyebrow    (10,      28),
;       upper hairline                  (11,      29),
;       lower hairline                  (12,      30),
;       face line                       (13,      31),
;       darkness of hair                (14,      32),
;       hair slant                      (15,      33),
;       nose line                       (16,      34),
;       size of mouth                   (17,      35),
;       curvature of mouth              (18,      36).
; -----------------------------------------------------------------------
; Usage       grface(obsrv, q, fl)
; Input     
;  Parameter  obsrv
;  Definition n x p matrix with the observations
;  Parameter  q
;  Definition 1 x 36 matrix with the assignments to the face parts
;  Parameter  fl
;  Definition scalar, faces per line
; Output    
;  Parameter  draw
;  Definition graphical object which contains the Flury faces 
; -----------------------------------------------------------------------
; Example
;    library("graphic")
;    q=0.*matrix(1,36)   ; do not use any facepart
;    q[,1] =1            ; right eye size
;    q[,19]=1            ; left eye size
;    q[,2] =2            ; right pupil size
;    q[,20]=2            ; left pupil size 
;    q[,4] =3            ; right eye slant
;    q[,22]=3            ; left eye slant
;    q[,11]=4            ; right upper hair line
;    q[,29]=4            ; left upper hair line
;    q[,12]=5            ; right lower hairline
;    q[,30]=5            ; left lower hairline
;    q[,13]=6            ; right face line
;    q[,14]=6            ; right darkness of hair
;    q[,31]=6            ; left face line
;    q[,32]=6            ; left darkness of hair
;    h=read("bank2")
;    h=h[91:110,]
;    draw=grface(h,q,5)
;    d=createdisplay(1,1)
;    axesoff()
;    show(d,1,1,draw)     
; -----------------------------------------------------------------------
; Result    Plot of the Flury faces
; -----------------------------------------------------------------------
; Author    Fabian Noetzel, Sigbert Klinke
; -----------------------------------------------------------------------
  error(prod(dim(p))<>36, "Vector p must have 36 entries")
  nx=countNotNumber(obsrv)
  error(sum(nx)>0, "Missings or infinite values")
  scal = sum((min(obsrv)<0)+(max(obsrv)>1), 2)
  if (scal)
    error (rows(obsrv)==1, "grface: can not scale only one observation")
    obsrv=(obsrv-min(obsrv))./(max(obsrv)-min(obsrv))
  endif
  width=1
  height=1
  rowobs=rows(obsrv)
  colobs=cols(obsrv)
  err=1
  i=0
  os=getenv("os")
  while (err==1)
    nmax=2000+i*1000
    xy=matrix(rowobs*nmax,3)*NaN
    pm=matrix(1,rowobs*nmax)*NaN
    sz1=1
    sz2=2  
    if (os=="windows")
      hndl = dlopen("face.dll")
      err = dlcall(hndl, "_faces_dlcall", obsrv, colobs, rowobs, p, width, height, xy, pm, sz1, sz2, nmax)
    else  
      hndl = dlopen("face.so")
      err = dlcall(hndl, "faces_dlcall", obsrv, colobs, rowobs, p, width, height, xy, pm, sz1, sz2, nmax)
    endif
    dlclose(hndl)
    error(err==22, "grface: vector p outside of the range [0,35]")
    error(err==10300, "grface: not enough memory") 
    i=i+1
  endo
  xym=xy[1:sz1]
  right=xym[,3]%fl
  down=(xym[,3]-right)/fl
  draw=(0.01*xym[,1]+right)~(0.01*xym[,2]+down)
  pm=pm[1,1:sz2]
  color = 0                          ;set color of line to blue 
  art = 1                          ;to draw a solid line 
  thick = 1                          ;to draw a relatively thick line
  setmaskp(draw,0,0,0) 
  setmaskl(draw, pm, color ,art ,thick) ;call setmaskl 
endp
