proc (alpha, sigma) = hillgp1 (x, k)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      pickandsgp mlegp mlegp0 dpgp momentgp hillgp1diag
; -----------------------------------------------------------------
;   Macro         hillgp1
; -----------------------------------------------------------------
;   Description   Hill estimator for GP1 model.
; -----------------------------------------------------------------
;   Usage         {alpha, sigma} = hillgp1 (x, k)
;   Input
;     Parameter   x
;     Definition      vector
;     Parameter   k
;     Definition      integer, number of upper extremes
;   Output
;     Parameter   alpha
;     Definition      scalar, estimated shape parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0)
;	            x=randx("gp1",100,1)
;             m=hillgp1(x,50)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.alpha
;             [1,]   0.96147
;             Contents of m.sigma
;             [1,]  0.85767 
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
  n0 = rows (x)
  x = log (sort (paf (x, x > 0)))
  n = rows (x)
  error (n < 2, "Data set must have at least 2 positive elements")
  if (k < 2)
    warning (k < 2, "Hill estimator requires at least two exceedances")
    k = 2
  endif
  if (k > n)
    warning (k > n, "k is larger than the number of positive data points")
    k = n
  endif
  alpha = 1 / (mean (x [n-k+2:n]) - x [n-k+1])
  sigma = exp (x [n-k+1]) * (k/n0)^(1/alpha)
endp
