proc(gest)=intest2d(t,y,h,g,loc,opt)
; -----------------------------------------------------------------
; Library     gam
; -----------------------------------------------------------------
;  See_also       intestpl, gintest, gintestpl, pcad, gamfit
; -----------------------------------------------------------------
;   Macro         intest2d
; -----------------------------------------------------------------
;   Description   estimation of a bivariate joint influence function
;                 and its derivatives in a model with possible interaction.
;                 When loc.lin.smoother is chosen you get the function
;                 estimate and the first derivatives in the first and
;                 second direction, when loc.quadr.smoother is chosen
;                 you get the function and the mixed derivative estimate.
; -----------------------------------------------------------------
;   Usage         gest = intest2d(t,y,h,g,loc{,opt})
;   Input
;     Parameter   t
;     Definition      n x p matrix , the observed explanatory variable
;                     where the directions of interest have to be the
;                     first and second column
;     Parameter   y
;     Definition      n x q matrix , the observed
;                     response variables
;     Parameter   h
;     Definition      2 x 1 or 1 x 1 matrix , chosen bandwidth for
;                     the directions of interest
;     Parameter   g
;     Definition      p x 1 or 1 x 1 matrix , chosen bandwidth for
;                     the directions not of interest
;     Parameter   loc
;     Definition      dummy, for loc=0 local constant (Nad. Wats.),
;                     for loc=1 local linear and for loc=2 local
;                     quadratic estimator will be used
;     Parameter   opt.tg
;     Definition      ng x 2 vector, a grid for direction of interest.
;                     If tg is given, the nonparametric function will be
;                     computed on this grid.
;     Parameter   opt.shf
;     Definition      integer,  (show-how-far) if exists and =1, an output
;                     is produced which indicates how the iteration
;                     is going on (additive function / point of estimation /
;                     number of iteration).
;   Output
;     Parameter   gest
;     Definition      n(ng) x pp x q matrix, containing the marginal
;                     integration estimates
; -----------------------------------------------------------------
;   Example   library("gam")
;             randomize(12345)
;             t     = grid(#(-0.9,-0.9),#(0.2,0.2),#(10,10))
;             n     = rows(t)
;             t     = t~(uniform(n)*2-1)
;             g3    = sin(2*t[,3])
;             g12   = t[,1].*t[,2]^2
;             y     = g3 + g12 + normal(n)*sqrt(0.5)
;             h     = #(1.0, 1.0)
;             g     = #(1.1, 1.1, 1.2)
;             loc   = 1
;             gest  = intest2d(t,y,h,g,loc)
;             library("graphic")
;             pic  = createdisplay(1,2)
;             dat11 = grsurface(t[,1:2]~g12)
;             dat12 = grsurface(t[,1:2]~gest[,1])
;             gc = grcube( dat11|dat12 )
;             show(pic,1,1,dat11,gc.box,gc.x,gc.y,gc.z,gc.c)
;             show(pic,1,2,dat12,gc.box,gc.x,gc.y,gc.z,gc.c)
;             setheadline(pic, 1, 1, "Original function")
;             setheadline(pic, 1, 2, "Estimated function")
; -----------------------------------------------------------------
;   Result    the marginal integration estimates of the joint
;             influences, using local polynomials,
;             see Sperlich, Tjostheim and Yang (1997)
; -----------------------------------------------------------------
;   Author    Sperlich   970716
; -----------------------------------------------------------------
 if (comp("opt","shf")>0)
   shf = opt.shf
 else
   shf = 0
 endif
 if (comp("opt","tg")<1)
   tg = t[,1:2]
 else
   tg = opt.tg
 endif
   error(cols(tg)<>2,"tg must have exactly 2 columns")
 p = cols(t)
 n = rows(t)
 q = cols(y)
 ng = rows(tg)
 o = sort(t[,1]~(1:n))[,2]
 ro = sort(o~(1:n))[,2]
 ys = y[o]
 ts = t[o]
 error(sum(loc==#(0,1,2))!=1,"loc is not correctly specified")
 if (rows(g)<>p)             /* reshape g to matrix */
   errtext = "rows of g didn't fit: "+string("%1.4f",g[1,1])+" assumed"
   warning(rows(g)>1,errtext)
   g = matrix(p)*g[1,1]
 endif
 if (rows(h)<>2)            /* reshape h to matrix */
   errtext = "rows of h didn't fit: "+string("%1.4f",h[1,1])+" assumed"
   warning(rows(h)>1,errtext)
   h = matrix(2)*h[1,1]
 endif
 if (loc==0)
   gest = matrix(ng,1,q)
   ys = ys~matrix(n)
  if (p>2)
     m = 1
     hs = g
     hs[1:2] = h
     while (m<=ng)
       if (shf)
         ("intest2d: "+string("%1.0f",1)+" "+string("%1.0f",m))
       endif
       test = ts
       test[,1:2] = tg[m,].*matrix(n)
       r = sker(ts,hs',"qua",ys,test)
       r = paf(r,r[,q+1].<>0)
       mh = r[,1:q]./r[,q+1]
       gest[m,1,] = mean(mh,1)
       m = m + 1
     endo
  else
       r = sker(ts,h',"qua",ys)[ro,]
       gest[,1,] = r[,1:q]./r[,q+1]
  endif
 else
   loc = loc-1               /* needed for locpol */
   if (p>2)
     hs = g
     hs[1:2] = h
     if (loc)
       gest = matrix(ng,2,q)*0
       posgest = 1 | 2
       posmb = 1 | (1+p+2)
     else
       gest = matrix(ng,3,q)*0
       posgest = 1 | 2
       posmb = 1 | 2 | 3
     endif
     m = 1
     while (m.<=ng)
       if (shf)
         ("intest2d: "+string("%1.0f",1)+" "+string("%1.0f",m))
       endif
       test = ts
       test[,1:2] = tg[m,].*matrix(n)
       ma = locpol(ts,test,ys,hs,loc)
       k = 1
       while (k<=q)
         sel = isNumber(ma[,1,k])
         if (sum(sel,1)>10)
           mb = paf(ma[,,k],sel)
         endif
         gest[m,posgest,k] = mean(mb[,posmb],1)
         k = k+1
       endo
       m = m+1
     endo
   else
       gest[,posgest,] = locpol(ts,ts,ys,h,loc)[ro,posmb,]
   endif
 endif
endp
