proc(j)=jump(data ,alpha)
; --------------------------------------------------------------------
; Library      times
; --------------------------------------------------------------------
; See_also     dlopen, dlcall, dlclose
; --------------------------------------------------------------------
; Macro        jump
; --------------------------------------------------------------------
; Description  detects jump points in the time series.
;              Optional parameter alpha controls the
;              the sensitivity of the procedure. Recomended
;              values are 0.5 ... 4.0. The default value is 2.0.
;              The output vector j, which has the same length 
;              as data, indicates the detected jumps.
;              NaN values are not allowed.
; ----------------------------------------------------------------
; Usage        j = jump(data {,alpha})   
; Input
;     Parameter data 
;     Definition n x 1 matrix
;     Parameter alpha
;     Definition 1x1 matrix (scalar) 
; Output
;     Parameter  j
;     Definition n x 1 matrix
; ----------------------------------------------------------------- 
; Example
;       library("times")   ; load library 
;       length = 200       ; size of the data
;       t=1:length
;       greed = t/length   ; greed on the interval [0,1]
;       vr = 0.3          ; variance of the white noise  
;       ;create a signal on the unit interval with 
;       ;two jump points at the 0.4 and 0.7
;       signal = 0.5.*greed.*(greed <= 0.4)+(0.4+0.2.*greed).*(greed>0.4)
;       signal = signal + 0.4.*(greed>0.7)
;       randomize(123456789012)
;       data = signal+ vr.*normal(length) ;add white noise to the signal
;       alpha = 4; 
;       jm = jump(data,alpha)
;       ; plot the data 
;       ; the observable data will be shown by the blue crosses
;       ; red line corresponds to the signal
;       ; magenta line shows the locations of the detected jumps 
;       signal=t~signal
;       jm= t~jm
;       data = t~data
;       setmaskl(signal,t',4,1,0)
;       setmaskp(signal,0,0,1)
;       setmaskl(jm,t',5,1,2)
;       setmaskp(jm,0,0,1)
;       setmaskl(data,t',0,2,1)
;       setmaskp(data,1,11,8)
;       output = createdisplay(2,1)
;       show(output,1,1,data)
;       show(output,2,1,jm,signal)
;       setgopt(output,2,1,"title","signal and detected jumps")
;       setgopt(output,1,1,"title","noisy signal")
; ------------------------------------------------------------------
; Result
;       the display "output" 
;---------------------------------------------------------------
; Keywords   
;       jump detection
;------------------------------------------------------------------
; Author     Golubev Y.  980609
; -----------------------------------------------------------------
row = rows(data)
j   = 1:row
if(exist("alpha")==0)
  alpha = 2.0
endif
if(alpha>=row)
 alpha=row/2
endif
if(alpha<= 0.000001)
  alpha = 0.000001
endif
;the source C++ code jmp.C was compiled by 
;gcc -G -O6 jmp.C -o jmp.so -lm
if (getenv("os")=="windows")
  h=dlopen("jmp.dll")
  dlcall(h,"_jmp",data,row,alpha,j)
  dlclose(h)
else
  h=dlopen("jmp.so")
  dlcall(h,"jmp",data,row,alpha,j)
  dlclose(h)
endif
endp
