proc(filtY) = kfilter(y,mu,Sig,H,F,Q,R)

; -----------------------------------------------------------------
; Library        times
; -----------------------------------------------------------------
;  See_also      ksmoother, kem
; -----------------------------------------------------------------
;   Macro        kfilter
; -----------------------------------------------------------------
;   Description  Calculates a filtered time serie (uni- or 
;                multivariate) using the Kalman filter equations.
;                The state-space model is assumed to be in the
;                following form:
;
;                y_t = H x_t + v_t
;
;                x_t = F x_t-1 + w_t
;
;                x_0 ~ (mu,Sig), v_t ~ (0,Q), w_t ~ (0,R)
;
;                All parameters are assumed known.
;
; -----------------------------------------------------------------
;   Keywords     Kalman filter, Kalman smoother, filter
; -----------------------------------------------------------------
;   Usage         fy = kfilter(y,mu,Sig,H,F,Q,R)    
;   Input
;     Parameter   y  
;     Definition      T x m matrix of observed time series, 
;                     T is the number of observations,
;                     m is the dimension of time series
;     Parameter   mu    
;     Definition      n x 1 vector, the mean of the initial state
;     Parameter   Sig    
;     Definition      n x n covariance matrix of the initial state
;     Parameter   H    
;     Definition      m x n matrix
;     Parameter   F    
;     Definition      n x n matrix
;     Parameter   Q   
;     Definition      m x m variance-covariance matrix
;     Parameter   R
;     Definition      n x n variance-covariance matrix 
;   Output                                                           
;     Parameter   fy
;     Definition      T x p matrix of filtered time series
; -----------------------------------------------------------------
;   Example   library("xplore")
;             library("times")
;             library("plot")
;
;             serie = read("kalman1.dat")
;             y = serie[,2]
;             mu = 10
;             Sig = 0
;             H = 1
;             F = 1
;             Q = 9
;             R = 9
;
;             fy = kfilter(y,mu,Sig,H,F,Q,R)
;
;             fserie = serie[,1]~serie[,2]~fy                        
;
;             data = fserie[,1]~fserie[,2]
;             data = setmask(data, "line", "red", "thin")
;             fdata = fserie[,1]~fserie[,3]
;             fdata = setmask(fdata, "line", "blue", "thin")
;
;             disp = createdisplay(1,1)
;             show(disp,1,1, data, fdata)
;             setgopt(disp,1,1, "title", "Kalman filter 1")
; -----------------------------------------------------------------
;   Result    Original serie is displayed with red colour,
;             filtered serie is displayed with blue colour.
;             (y is a lagged random walk with errors.)
; -----------------------------------------------------------------
;   Example   library("xplore")
;             library("times")
;             library("plot")
;
;             serie = read("kalman3.dat")
;             y  = serie[,2:3] 
;             mu = #(20,0)
;    
        Sig = #(0,0)~#(0,0)
;             H  = #(0.3,-0.3)~#(1,1)
;             F  = #(1,0)~#(1,0)
;             Q  = #(9,0)~#(0,9)
;             R  = #(0,0)~#(0,9)
;
;             fy = kfilter(y,mu,Sig,H,F,Q,R)
;
;             fserie = serie[,1]~serie[,2]~serie[,3]~fy[,1]~fy[,2]
;
;             data1 = fserie[,1]~fserie[,2]
;             data1 = setmask(data1, "line", "red", "thin")
;
;             fdata1 = fserie[,1]~fserie[,4]
;             fdata1 = setmask(fdata1, "line", "blue", "thin")
;
;             data2 = fserie[,1]~fserie[,3]
;             data2 = setmask(data2, "line", "red", "thin")
;
;             fdata2 = fserie[,1]~fserie[,5]
;             fdata2 = setmask(fdata2, "line", "blue", "thin")
;
;             disp = createdisplay(2,1)
;             show(disp,1,1, data1, fdata1)
;             setgopt(disp, 1, 1, "title", "Kalman filter 2 - 1st element")
;             show(disp,2,1, data2, fdata2)
;             setgopt(disp,2,1, "title", "Kalman filter 2 - 2nd element")
; -----------------------------------------------------------------
;   Result    Original serie is displayed with red colour,
;             filtered serie is displayed with blue colour.
;             (y is a lagged bivariate MA process with errors.)
; -----------------------------------------------------------------
;   Example   library("xplore")
;             library("times")
;             library("plot")
;
;             serie = read("kalman2.dat")
;             y = serie[,2]
;             mu = #(0,0)
;            
Sig = #(0,0)~#(0,0)
;             H = #(1,0)'
;             F = #(0.5,1)~#(-0.3,0)
;             R = #(1,0)~#(0,0)
;             Q = 4
;             fy = kfilter(y,mu,Sig, H,F,Q,R)
;
;             fserie = serie[,1]~serie[,2]~fy
;
;             data1 = fserie[,1]~fserie[,2]
;             data1 = setmask(data1, "line", "red", "thin")
;
;             fdata1 = fserie[,1]~fserie[,3]
;             fdata1 = setmask(fdata1, "line", "blue", "thin")
;
;             disp = createdisplay(1,1)
;             show(disp,1,1, data1, fdata1)
;             setgopt(disp,1,1, "title", "Kalman filter 3")
; -----------------------------------------------------------------
;   Result    Original serie is displayed with red colour,
;             filtered serie is displayed with blue colour.
;             (y is an AR(2) process with errors.)
; -----------------------------------------------------------------
;   Author    P.Franek 990507
; -----------------------------------------------------------------
  
; as a by-product saves a matrix KFOutPut with all x_t|t and P_t|t
  
  dimX = rows(F)                                ;get dimension of state vector
  PreviousX = mu                                ;set starting state x_0|0 = mu
  PreviousP = Sig                               ;set starting covariance matrix P_0|0 = Sigma
  
  KFOutPut = vec(PreviousX,vec(PreviousP))
  
  T = rows(y)                                   ;get number of observations (T) 
  filtY = y .* 0                                ;prepare the result matrix
  
  i = 1                                         ;main loop
  while (i <= T)
    
; Step of the Kalman recursion - the notation follows Brockwell&Davis
    
    pom = (F * PreviousP * F' + R)              ;temporary variable, P_(t|t-1) in fact
    Delta = H * pom * H' + Q
    DeltaInv = ginv(Delta)
    Innovation = (y[i,]' - H * F * PreviousX)
    CurrentX = F * PreviousX + pom * H' * DeltaInv * Innovation     
    CurrentP = pom - pom * H' * DeltaInv * H * pom  
    
; Step of the loop 
    
    filtY[i,] = (H * CurrentX)'             ;note the transposition 
    PreviousX = CurrentX
    PreviousP = CurrentP 
    
    KFOutPut = KFOutPut~vec(PreviousX,vec(PreviousP))
    
    i = i + 1 
  endo
  KFOutPut = KFOutPut'
  write(KFOutPut,"KFOutPut.dat","x")
endp
