proc(beta,bse,bstan,bpval)=linreg(x,y,opt,om)
; -----------------------------------------------------------------------
; Library      stats
; -----------------------------------------------------------------
; See_also     gls linregfs linregres doglm
; -----------------------------------------------------------------------
; Macro        linreg
; -----------------------------------------------------------------------
; Description  linreg computes the Generalized Least Squares estimate for
;              the coefficients of a linear model.
; ----------------------------------------------------------------------
;   Notes         
;        We can specify a covariance matrix sigma^2 * om for the errors.
;        The estimate is given by b = INV(TRN(x) INV(om) x) TRN(x) INV(om) y.
;        The parameters opt and om are optional and can be omitted:
;	       opt can consist of "notext"|"intercept"|"display"|"omega".
;	       For opt you can use any combination of the above or none,
;	       however if you want to use a covariance matrix om then
;	       you must specify the option "omega".
;        Default is the output of an ANOVA table in the output 
;	       window of XploRe, which you can switch off by "notext",
; 	     in this case linreg only gives you back: b, standard error of b,
;	       b standardized and the respective p-values.
; -----------------------------------------------------------------------
; Usage        {b,bse,bstan,bpval} = linreg (x, y{,opt, om})
; Input
;   Parameter  x 
;   Definition n x p x d1 x ... x dn array    
;   Parameter  y 
;   Definition n x 1 x d1 x ... x dn array  
;   Parameter  opt
;   Definition string vector
;   Parameter  om
;   Definition n x n x d1 x ... x dn array   
; Output
;   Parameter  b
;   Definition p x 1 x d1 x ... x dn array 
;   Parameter  bse
;   Definition p x 1 x d1 x ... x dn array 
;   Parameter  bstan
;   Definition p x 1 x d1 x ... x dn array 
;   Parameter  bpval
;   Definition p x 1 x d1 x ... x dn array   
; -----------------------------------------------------------------------
; Example      library("stats")
;              setenv("outputstringformat", "%s")    
;              randomize(1964)
;              n = 500
;              x = normal(n,3)
;              beta = #(10, 2, 3)
;              u = 0.5 * normal(n) 
;   	         y = x*beta .+ u
;              {beta,se,betastan,p} = linreg(x,y)  
;-----------------------------------------------------------------------
; Result       Contents of out
;              [ 1,] 
;              [ 2,] A  N  O  V  A                   SS      df     MSS       F-test   P-value
;              [ 3,] _________________________________________________________________________
;              [ 4,] Regression                 61494.937     3 20498.312   80164.745   0.0000
;              [ 5,] Residuals                    126.828   496     0.256
;              [ 6,] Total Variation            61621.765   499   123.491
;              [ 7,] 
;              [ 8,] Multiple R      = 0.99897
;              [ 9,] R^2             = 0.99794
;              [10,] Adjusted R^2    = 0.99793
;              [11,] Standard Error  = 0.50567
;              [12,] 
;              [13,] 
;              [14,] PARAMETERS         Beta         SE         StandB        t-test   P-value
;              [15,] ________________________________________________________________________
;              [16,] b[ 0,]=         -0.0058       0.0227       0.0000        -0.254   0.6001
;              [17,] b[ 1,]=         10.0019       0.0215       0.9501       465.977   0.0000
;              [18,] b[ 2,]=          1.9906       0.0221       0.1839        90.263   0.0000
;              [19,] b[ 3,]=          3.0249       0.0231       0.2667       130.817   0.0000
;-----------------------------------------------------------------------
; Example      library("stats")
;              randomize(1964)
;              n = 50
;              x = normal(n,3)
;              beta = #(10, 2, 3)
;              u = 0.5 * normal(n) 
;              y = x*beta .+ u
;              covar = (0.5.*x[,2] .+ 0.3.*x[,3]+ 0.2*x[,1]).^2
;              y = x*beta .+ sqrt(covar).*u
;              {beta,se,betastan,p} = linreg(x, y,"nointercept"|"display"|"omega",diag(covar))   
; -----------------------------------------------------------------------
; Result 
; You see the display:
;         A  N  O  V  A                 SS       df     MSS      F-test     P-value
; 	     _________________________________________________________________________
;	       Regression                 4782.570     3  1594.190    8526.286   0.0000
;	       Residuals                     8.601    46     0.187
;	       Total Variation            4846.290    49    98.904
;	       Multiple R      = 0.99340
;	       R^2             = 0.98685
;	       Adjusted R^2    = 0.99811
;	       Standard Error  = 0.43240
;	       PARAMETERS         Beta        SE         StandB      t-test      P-value
;        _________________________________________________________________________
;	       b[ 1,]=         10.0297      0.0643       0.9793      155.865     0.0000
;	       b[ 2,]=          2.0544      0.0518       0.2452       39.661     0.0000
;	       b[ 3,]=          2.9838      0.0706       0.2673       42.291     0.0000  
; -----------------------------------------------------------------------
; Author       Susanne Hannappel   961122
; ----------------------------------------------------------------------
if (exist(opt)) //with_without omega and with_without intercept
	if (sum(opt=="omega")>0)	
		if (exist(om))
			if (sum(opt=="nointercept")>0)	
				beta =gls(x,y,om)
				intercept=0
			else
				x=matrix(rows(x))~x
				beta =gls(x,y,om)
				intercept=1
			endif  
		endif
		
	else	
		if (sum(opt=="nointercept")>0)	
			beta =gls(x,y)
			intercept=0
		else
			x=matrix(rows(x))~x
			beta =gls(x,y)
			intercept=1
		endif
	endif
else
	x=matrix(rows(x))~x
	beta =gls(x,y)
	intercept=1
endif //exist
// Output
{out,bse,bstan,bpval}=anova(x,y,beta,intercept)
if(exist(opt))
	if (sum(opt=="display")>0)
		
		if (sum(opt=="notext")<0.001)		
			di=createdisplay(1,1)
			show(di,1,1,out)
		endif	
	else
		if (sum(opt=="notext")<0.001)
			out
		endif 		
	endif 
else
        out
endif //exist opt
endp //linreg
proc(output,betase,betastan,pval)=anova(x,y,beta,int)
//ANOVA table
		
	yhat=x*beta
	ybar=mean(y)
	
	if (abs(int-1)<0.001)//with intercept
 		dfM=rows(beta)-1	
	else
		dfM=rows(beta)	
	endif
	 
	dfMstr=string("%6.0f",dfM)
	SSM= sum((yhat .-ybar).*(yhat .-ybar))
	SSMstr=string("%10.3f",SSM)
	MSSM=SSM/dfM
	MSSMstr=string("%10.3f",MSSM)
	
	dfR=rows(x)-1-dfM
	dfRstr=string("%6.0f",dfR)
	SSR= sum ((yhat - y).*(yhat - y))
	SSRstr=string("%10.3f",SSR)
	MSSR=SSR/dfR
	MSSRstr=string("%10.3f",MSSR)
		
	dfT=rows(x)-1
	dfTstr=string("%6.0f",dfT)
	SST= sum ((y - ybar).*(y - ybar))
	SSTstr=string("%10.3f",SST)
	MSST=SST/dfT
	MSSTstr=string("%10.3f",MSST)
	
	Fvalue=(SSM * dfR)/(SSR * dfM)
	Fvalstr=string("  %10.3f", Fvalue)
	FPvalue=1-cdff(Fvalue,dfM,dfR)
	FPvalstr=string("   %1.4f", FPvalue)
	
	MultipR= sqrt(SSM/SST)
	MultipRstr=string("Multiple R      = %7.5f",MultipR)
	Rsquare= SSM/SST
	Rstr=string("R^2             = %7.5f",Rsquare)
	adRsquare= 1-((dfT*SSR)/(dfR*SST))
	adRstr=string("Adjusted R^2    = %7.5f",adRsquare)
	StanEr= sqrt((sum((y-yhat).*(y-yhat))/dfR))
	StanErstr=string("Standard Error  = %7.5f",StanEr)
//PARAMETERS with t-test
	
	ttestmatrix=(inv(x'*x))	
	diagttestmatrix=xdiag(ttestmatrix)
	betase=sqrt(MSSR*diagttestmatrix)
	betasestr=string("     %8.4f",betase)
	xbar=mean(x)
	xstddev=sqrt(sum((x.-xbar).*(x.-xbar)))	
	ystddev=sqrt(sum((y-ybar).*(y-ybar)))	
	sksy=trans(xstddev./ystddev)	
	betastan=beta.*sksy
	betastanstr=string("     %8.4f",betastan)
	tvalues=beta./(sqrt(MSSR*diagttestmatrix))
	tvalstr=string("  %12.3f", tvalues) 
	 
;	tPvalues=1-cdft(tvalues,dfR)
        tPvalues=2*(1-cdft(abs(tvalues),dfR))
	pval=tPvalues
	tPvalstr=string("   %1.4f", tPvalues)
	if (abs(int-1)<0.001)//with intercept
 		betastr=string("b[%2.0f,]= %15.4f", 0:(rows(beta)-1), beta)	
	else
		betastr=string("b[%2.0f,]= %15.4f", 1:rows(beta), beta)	
	endif
//TEXTOUTPUT	
	 
	anfirstline= "A  N  O  V  A                   SS      df     MSS       F-test   P-value"
	ansecondline="_________________________________________________________________________"
	anovatextMod= "Regression                "+SSMstr+dfMstr+MSSMstr+Fvalstr+FPvalstr
	anovatextRes= "Residuals                 "+SSRstr+dfRstr+MSSRstr
	anovatextTot= "Total Variation           "+SSTstr+dfTstr+MSSTstr	
	anovatext=""|anfirstline|ansecondline|anovatextMod|anovatextRes|anovatextTot|""|MultipRstr|Rstr|adRstr|StanErstr
	
	parfirstline="PARAMETERS         Beta         SE         StandB        t-test   P-value"    
	parsecondline="________________________________________________________________________"
	parametertext=""| parfirstline|parsecondline|betastr+betasestr+betastanstr+tvalstr+tPvalstr
	output= anovatext|""|parametertext
endp
  
