proc(T) = lobrob(y,bdvec)
;----------------------------------------------------------------------
;  Library 	  times
;----------------------------------------------------------------------
;  See_also	  lo, kpss, rvlm, robwhittle, roblm, gph, sclmtest, pgram,
;----------------------------------------------------------------------
;  Macro	  lobrob
;----------------------------------------------------------------------
;   Description   Semiparametric test for I(0) of a time series against 
;     fractional alternatives, i.e., long-memory and antipersistence.
;     The test is semiparametric in the sense that it does not depend on 
;     a specific parametric form of the spectrum in the neighborhood of the 
;     zero frequency. 
;     The first argument of the function is the series. The second optional 
;     argument is the vector of bandwidth, i.e., the parameter specifying 
;     the number of harmonic frequencies around zero to be considered. 
;     By default, the macro uses the automatic bandwidth given in Lobato and 
;     Robinson. If the user provides his own vector of bandwidths, then the
;     function returns the value of the test for each component of the bandwidth vector.      
;     If the value of the test is in the lower tail of the standard
;     normal distribution, the null hypothesis of I(0) is rejected against the 
;     alternative that the series displays long-memory. If the value of the 
;     test is in the upper tail of the standard normal distribution, the null
;     hypothesis I(0) is rejected against the alternative that the series is
;     antipersistent.
; ----------------------------------------------------------------------
;   Usage         T = lobrob(y{,bdvec})
;   Input 
;     Parameter   y
;     Definition  vector
;     Parameter   bdvec
;     Definition  vector
;   Output
;     Parameter   T
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;Test for long-memory in the volatility. As there is no optional 
;               ;argument, the automatic bandwidth is used
;               library("times")
;               x = read("dmus58.dat")
;               x=x[1:1000]
;               y = abs(tdiff(x)) 
;               T = lobrob(y)
;               T
; ----------------------------------------------------------------------
;   Result      ; As T is negative, and in the lower tail of the standard normal 
;               ; distribution, there is evidence of long-memory in the volatility
;               ; The automatic bandwidth is equal to 301
;               Contents of T
;               [1,] Bandwidth   Statistic   
;               [2,] _____________________  
;               [3,]  
;               [4,]    301      -3.8377 
; 
; ----------------------------------------------------------------------
;   Example     ;The same as above, except that a 3-dimensional vector m of 
;		            ;bandwidth is provided as optional second argument.
;               library("times")
;               x = read("dmus58.dat")
;               x=x[1:1000]
;               y = abs(tdiff(x)) 
;               m = #(150,200,250)
;               T = lobrob(y,m)
;               T
; ----------------------------------------------------------------------
;   Result      ; T is negative, thus we also conclude that there is evidence for 
;               ; long-memory in the volatility for m = 150, 200, 250
;               Contents of T
;               [1,] Bandwidth   Statistic  
;               [2,] _____________________  
;               [3,]  
;               [4,]    150      -2.7637 
;               [5,]    200      -3.4422 
;               [6,]    250      -3.2086 
; ----------------------------------------------------------------------
;   Reference
;     Lobato, I. and P.M. Robinson (1997): A Nonparametric Test for I(0)
;      LSE STICERD Discussion paper EM/97/342
;----------------------------------------------------------------------
;   Keywords
;      long memory, tests, nonstationarity
;----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 981013
;----------------------------------------------------------------------
  opt = 0 ; tests the presence of the optional matrix of bandwidth parameters
  if (exist(bdvec)==1)
    opt=1   
  endif
  ;;
  n = rows(y)  ; This part of the code is from the macro pgram (Rong Chen, Christian Hafner, 1997)
  y = y-mean(y)
  p = log(n)/log(2)
  p = rint(exp(log(2)*ceil(p))) ; next power of 2 
  if (n != p)
    y = y|(matrix(p-n)-1)       ; append series 
  endif
  y = y~(matrix(p).*0)
  ftr = fft(y)                  ; fourier transform
  I = (sum((ftr.*ftr)')/p)'     ; periodogram
  ind=2:(floor(n/2))
  I = I[ind]
  ;;
  if (opt == 0)
    imax = 1
    mpo  = n^(0.8)
    mmoins = 0.06*mpo
		mplus = 1.2*mpo
    phiat = acf(y[,1])[2]
    mdstar = (3*n/(4*pi))^(0.8)*(abs(-phiat/(1-phiat)^2))^(-0.4)
    if (mdstar < mmoins)
      range = #(mmoins)
    else
      if (mdstar  < mplus)
        range = #(mdstar)
      else
        range = #(mplus)
      endif
    endif
  else
    imax =  rows(bdvec)
    range = bdvec
  endif 
  LMT = matrix(imax)
  mbandwidth = rint(range)
  i = 1
  while (i <= imax)
    error(range[i] > n/2,"Bandwidth greater than the Nyquist frequency")      
    VJ = log(cumsum(matrix(mbandwidth[i])))
    VJ = VJ-mean(VJ)
    IJ = I[1:mbandwidth[i]]
    C  = mean(IJ)
    C1 = mean(IJ .* VJ)
    LMT[i] = sqrt(mbandwidth[i])*C1/C
    i = i+1  
  endo
  ;;
  ;;T = LMT
  ; Output display
  line1 ="Bandwidth   Statistic "
  line2 ="_____________________ "
  st1 = string("%6.0f",mbandwidth)
  st2 = string("%13.4f",LMT)
  st3 = st1+st2
  T = line1|line2|""|st3  
endp
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro lobrob
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



