proc (mh)=lregestp(x,h,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regestp lpregest lregxestp
; -----------------------------------------------------------------
;   Macro        lregestp
; -----------------------------------------------------------------
;   Description  estimates a multivariate regression function using
;                local polynomial kernel regression. The computation
;                uses WARPing.
; -----------------------------------------------------------------
;   Notes        The WARPing enhances the speed of computation,
;                but may lead to computational errors, if the
;                bandwidth is small. For p<=2 WARPing is usually
;                faster than exact computation. For p>2, the
;                macro "regxestp" should be used instead.
; -----------------------------------------------------------------
;   Reference    Local Polynomial Fitting, Ruppert/Wand (1995)
;
;                Binning for local polynomials, Fan/Marron (1994)
;
;                WARPing method, W. Haerdle, "Smoothing Techniques
;                with applications in S"
; -----------------------------------------------------------------
;   Usage        mh = lregestp(x {,h {,K {,d}}})
;   Input
;     Parameter  x
;     Definition   n x (p+1), the data. In the first p columns the
;                  independent, in the last column the dependent
;                  variable.
;     Parameter  h
;     Definition   scalar or p x 1 vector, bandwidth. If not
;                  given, 20% of the volume of x[,1:p] is used.
;     Parameter  K
;     Definition   string, kernel function on [-1,1]^p. If not given,
;                  the product Quartic kernel "qua" is used.
;     Parameter  d
;     Definition   scalar, discretization binwidth. d[i] must be
;                  smaller than h[i]. If not given, the minimum of h/3
;                  and (max(x)-min(x))'/r, with r=100 for p=1, and
;                  r=(1000^(1/p)) for p>1 is used.
;   Output
;     Parameter  mh
;     Definition   m x (p+1) matrix, the first p columns constitute
;                  a grid and the last column contains the regression
;                  estimate on that grid.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             library("plot")
;             ;
;             x = 4.*pi.*(uniform(400,2)-0.5)
;             m = sum(cos(x),2)
;             e = uniform(400)-0.5
;             x = x~(m+e)
;             ;
;             mh = regestp(x,2)
;             mh = setmask(mh, "surface","blue")
;             m  = setmask(x[,1:2]~m,"black","cross","small")
;             plot(mh,m)
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The Local Linear regession estimate (blue) using
;             Quartic kernel and bandwidth h=2 and the true
;             regression function (thin black crosses) are pictured.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990419
; -----------------------------------------------------------------
  error(cols(x)<2,"lregestp: cols(x) < 2")
  n=rows(x)
  p=cols(x)-1
  ix=min(x[,1:p])
  mx=max(x[,1:p])
  rx=mx-ix
;
  if (exist(h)==0)
    h=((0.2)^(1/p)).*rx
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (rows(h)==1)
    h = h.*matrix(p)
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
;
  if (exist(d)==0)
    r=100.*(p==1)+(1000^(1/p)).*(p>1)
    d=(max(x[,1:p])-min(x[,1:p]))'/r
    d=min(d~(h./3),2)
  endif
  if (cols(d)>1)
    d=d'
  endif
  if (rows(d)==1)
    d = d.*matrix(p)
  endif
;
  error(sum(h .<=d)>0,"lregestp: h smaller than d -- Please modify h or d!")
;
  {xb,yb}=bindata(x[,1:p],d,0.*matrix(p),x[,p+1])
;
  m=(max(xb)-min(xb)+1) '
  org=matrix(p)-1
  dh=d./h
  hd=h./d
  g=grid(org,dh,hd)
  wx=grid(org,matrix(p),hd)
  wy0=_K(g)
;
  {xc,c,or}=conv(xb,yb,wx,wy0)             ; S0
  m=rows(c)
  sx=matrix(p+1,p+1,m)-1
  tx=matrix(p+1,1,m)-1
  sx[1,1]=reshape(c[,1],1|1|m)
  tx[1,1]=reshape(c[,2],1|1|m)
;
  bin=2^(0:(p-1))
;
  j=0
  while (j<p)
    j=j+1
    wy=wy0.*g[,j]
    sym=bin[j]
    {xc,c,or}=conv(xb,yb,wx,wy,sym)        ; S1, T1
    sx[1,j+1]=reshape(c[,1],1|1|m)
    tx[j+1,1]=reshape(c[,2],1|1|m)
    sx[j+1,1]=sx[1,j+1]
  endo
;
  j=0
  while (j<p)
    j=j+1
    k=0
    while (k<j)
      k=k+1
      wy=wy0.*g[,j].*g[,k]
      sym=(j!=k).*(bin[j]+bin[k])
      {xc,c,or}=conv(xb,yb[,1],wx,wy,sym)  ; S2
      sx[j+1,k+1]=reshape(c,1|1|m)
      sx[k+1,j+1]=reshape(c,1|1|m)
    endo
  endo
;
  mh=inv(sx)*tx
  mh=((xc+0.5).*d')~reshape(mh[1,1],m|1|1)
  mh=sort(mh,1:p)
endp