proc(output)=pandyn2(z,p,IVmeth,beta,T) 
; ---------------------------------------------------------------------
; Library     metrics
; ---------------------------------------------------------------------
; See_also    panfix, pandyn
; ---------------------------------------------------------------------
; Macro       pandyn2
; ---------------------------------------------------------------------
; Description 2nd Step of the 2-stage dynamic GMM 
; ---------------------------------------------------------------------
; Usage       (output)=pandyn2(z,p,IVmeth,beta {,T})
; Input       
; Parameter   z       
; Definition          NT x {2}+1+k  matrix
; Parameter   p       
; Definition          Autoregressive order for dependent variable
; Parameter   IVmeth   
; Definition          Indicator for the instruments. 
;                     IVmeth = 0 lets the program choose the IV-method
; Parameter   beta    
; Definition          (k+p) x 1  vector of coefficient estimates
; Parameter   T       
; Definition          Common time periods for balanced panel (optional)
; Output      
; Parameter   output  
; Definition          Output table (string)
; ---------------------------------------------------------------------
; Notes       See Arellano and Bond (1991) for a method description
; ---------------------------------------------------------------------
; Example     z=read("dynpanel.dat")   ; read data set
;             {output,b}=pandyn(z,1,0) ; compute first step
;             pandyn2(z,1,0,b)         ; computes the second step
; ---------------------------------------------------------------------
; Result      
; [ 1,] "====================================================="
; [ 2,] "2-Stage-GMM for the Dynamic Fixed-Effect Model:     "
; [ 3,] "y(i,t) = y(i,t-1)'gamma + x(i,t)'beta + a(i) + e(i,t)"
; [ 4,] "====================================================="
; [ 5,] "PARAMETERS        Estimate        SE          t-value"
; [ 6,] "====================================================="
; [ 7,] "beta[ 1 ]=          0.9811       0.2971         3.302"
; [ 8,] "beta[ 2 ]=         -1.1420       0.2288        -4.991"
; [ 9,] "Lag [ 1 ]=          0.8262       0.1035         7.983"
; [10,] "====================================================="
; [11,] "N*T=    375      N=   25      DF=     2              "
; [12,] "R-square (differenced model):    0.5998              "
; [13,] "Hansen's J-statistic (p-val):    0.0441 (0.9782)     "
; [14,] "Instruments according to method: 1                   "
; [15,] "====================================================="
;  ---------------------------------------------------------------------
; Keywords    GMM, dynamic panel data model
; ---------------------------------------------------------------------
; Reference   Arellano and Bond (1991)
; ---------------------------------------------------------------------
; Link        
; ---------------------------------------------------------------------
; Author      Joerg Breitung (990316)
; ---------------------------------------------------------------------

  NT=rows(z)   
  k=cols(z)  
  if (exist(T)) 
    Tmin=1
    Tmax=T
  else  
    Tmin=min(z[,2])
    Tmax=max(z[,2])
  endif

  zy=0
  zx=0
  yy=0
  xy=0
  xx=0
  sx=0
  sy=0
  sxx=0
  sxy=0
  ssy=0
  ssx=0
  syy=0
  ZuuZ=0
  sT=0

                          ; balanced case
  if (exist(T))  
    N=NT/T
    error(N-floor(N)>0,"  The data set is not balanced (NT/N is not an integer)")
    error(T<2,         "  T must be at least 2")
    T0=T-p-1
      error(T0<2,"  T must be at least p+2")
    range0=T0
    range1=T0
    if (IVmeth==0)
      ivn1=T0*(T0+1)/2 + (k-1) + N/10
      ivn2=T0 + T0*(k-1) + N/10
      ivn3=T0+k-1+N/10
      IVmeth=4
      if (ivn1>N)
        IVmeth=3
        if (ivn2>N)
           IVmeth=2
           if (ivn3>N)
              IVmeth=1
           endif
        endif
      endif   
    endif

                        
    i=1 
    while (i<=N)
      i0=(i-1)*T+1
      i1=i*T
      yi=z[i0:i1,1] 
      xi=z[i0:i1,2:k] 
      dxi=xi[p+2:T,]-xi[p+1:T-1,]

      if (IVmeth==1) 

        zi=yi[p:T-2]
        j=2
        while (j<=p)
          zi=zi~yi[p-j+1:T-j-1]
          j=j+1
        endo
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]
        zi=zi~(xi[p+1:T-1,]-xi[p:T-2,])~dxi
        zi=zi'
      endif

      if (IVmeth==2) 

        zi=(matrix(T0,1)*yi[p])
        j=p+1
        while (j<=T-2)
          z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
          zi=zi~z0
          j=j+1
        endo
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]
        zi=zi|(dxi')
      endif

      if (IVmeth==3)  

        zi=(matrix(T0,1)*yi[p])
        j=p+1
        while (j<=T-2)
          z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
          zi=zi~z0
          j=j+1
        endo 
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]

        ein=unit(T0)
        z0=matrix((k-1)*T0,T0)
        j=1
        while (j<=T0)
          z0[,j]=kron(ein[,j],(dxi[j,]'))
          j=j+1
        endo
        zi=zi|z0 
      endif 

      if (IVmeth==4)

        zi=yi[1:p]~(0*matrix(p,T0-1))  
        j=p+1
        while (j<=T-3)
          z0=yi[1:j]~(0*matrix(j,T-j-2))
          z0=(0*matrix(j,j-p))~z0
          zi=zi|z0
          j=j+1
        endo
        z0=(0*matrix(T-2,T-2-p))~yi[1:T-2]
        zi=zi|z0
        zi=zi|(dxi')
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]
      endif


      j=1
      while (j<=p)
        dxi=dxi~(yi[p+2-j:T-j,]-yi[p+1-j:T-j-1,])
        xi=xi[p+1:T,]~yi[p-j+1:T-j]
        j=j+1
      endo

      y=yi[p+1:T]
      x=xi
      sxy=sxy+dxi'*dyi
      syy=syy+dyi'*dyi
      sxx=sxx+dxi'*dxi
      sy=sy+sum(dyi)
      sx=sx+sum(dxi)'

      zy=zy+zi*dyi 
      zx=zx+zi*dxi
      yy=yy+dyi'*dyi
      xy=xy+dxi'*dyi
      xx=xx+dxi'*dxi
      Zu=zi*(dyi-dxi*beta)
      ZuuZ=ZuuZ+Zu*Zu'
      sT=sT+T0
      
    i=i+1 
    endo

    k=k-1
    
   else 

                           ; unbalanced case
     T=Tmax-Tmin+1
     T0=T-p-1
     N=NT/T 
     range0=Tmax
     range1=Tmin
     if (IVmeth==0)
       ivn1=T0*(T0+1)/2 + (k-3) + N/10
       ivn2=T0 + T0*(k-3) + N/10
       ivn3=T0+k-3+N/10
       IVmeth=4
       if (ivn1>N)
         IVmeth=3
         if (ivn2>N)
            IVmeth=2
            if (ivn3>N)
              IVmeth=1
           endif
         endif
       endif   
     endif
    
     pos=1
     N=0
     while (pos < NT-2) 
       yi=0*matrix(1,1)
       xi=0*matrix(1,k-3)    
       dyi=0*matrix(2,1) 
       dxi=0*matrix(2,k+p-3)
       ctl=dyi
       dvec=0*matrix(T,1)
       Ti=0
       year=Tmin
       while (year<=Tmax)
         if (z[pos,2]==year)
           dvec[year-Tmin+1]=1 
           yi=yi|z[pos,3]
           xi=xi|z[pos,4:k]
           pos=pos+1
         else  
           yi=yi|(0*z[pos,3])
           xi=xi|(0*z[pos,4:k])
         endif
         
         if (year>=Tmin+p+1)
           d=0
           m=rows(yi)
           if (sum(dvec[m-p-2:m-1])==p+2)           
              d=1
           endif
           ctl=ctl|d
           dyi=dyi|(d*(yi[m]-yi[m-1]))
           dxi=dxi|(d*((xi[m,]-xi[m-1,])~(yi[m-1:m-p]-yi[m-2:m-p-1])' ))
           Ti=Ti+d
         endif
  
       year=year+1
       endo

       yi=yi[2:T+1]
       xi=xi[2:T+1,]
       dyi=dyi[3:rows(dyi)]
       dxi=dxi[3:rows(dxi),]
       ctl=ctl[3:rows(ctl)]
                                       
       if (IVmeth==1)  

         zi=yi[p:T-2] 
         j=2
         while (j<=p)
           zi=zi~yi[p-j+1:T-j-1]
           j=j+1
         endo
         zi=zi~(xi[p+1:T-1,]-xi[p:T-2,])~dxi[,1:k-3] 
         zi=zi'
       endif

       if (IVmeth==2)   

         zi=(matrix(T0,1)*yi[p])
         j=p+1
         while (j<=T-2)
           z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
           zi=zi~z0
           j=j+1
         endo

         zi=zi|(dxi[,1:k-3]')
       endif

       if (IVmeth==3)   

         zi=(matrix(T0,1)*yi[p])
         j=p+1
         while (j<=T-2)
           z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
           zi=zi~z0
           j=j+1
         endo

         ein=unit(T0)
         z0=matrix((k-3)*T0,T0)
         j=1
         while (j<=T0)
           z0[,j]=kron(ein[,j],(dxi[j,1:k-3]'))
           j=j+1
         endo
         zi=zi|z0 
       endif

       if (IVmeth==4)   

         z0=yi[1:p]~(0*matrix(p,T-p-2))
         zi=z0   
         i=p+1
         while (i<=T-3) 
           z0=yi[1:i]~(0*matrix(i,T-i-2))
           z0=(0*matrix(i,i-p))~z0
           zi=zi|z0
           i=i+1
         endo
         z0=(0*matrix(T-2,T-2-p))~yi[1:T-2]
         zi=zi|z0
         zi=zi|(dxi[,1:k-3]')
       endif
         
       Ti=Ti-1      
       range0=min(range0|Ti)
       range1=max(range1|Ti)
       y=yi[p+1:T]
       x=xi[p+1:T,]      
       zi=(zi'.*ctl)' 
       sxx=sxx+dxi'*dxi
       sxy=sxy+dxi'*dyi
       syy=syy+dyi'*dyi
       sy=sy+sum(dyi)
       sx=sx+sum(dxi)' 
       zy=zy+zi*dyi  
       zx=zx+zi*dxi
       Zu=zi*(dyi-dxi*beta)
       ZuuZ=ZuuZ+Zu*Zu'
       yy=yy+dyi'*dyi
       xy=xy+dxi'*dyi
       xx=xx+dxi'*dxi
       sT=sT+sum(ctl)
       N=N+1
     endo       
     k=k-3
   endif 


;     error(rows(zwz)>N,"  number of instruments exceeds N")
   zwz=inv(ZuuZ)
   bgmm=inv(zx'*zwz*zx)*zx'*zwz*zy
   sige=yy-2*bgmm'*xy+bgmm'*xx*bgmm
   sige=sige/2/sT
   SEbgmm=sige*inv(zx'*zwz*zx) 
   SEbgmm=sqrt(xdiag(SEbgmm))	

;                      computation of R2 as correlation coefficient

   cov=bgmm'*(sxy-sx*sy/sT)/sT
   var1=bgmm'*(sxx-sx*sx'/sT)*bgmm/sT
   var2=(syy-sy^2/sT)/sT
   R2=cov^2/var1/var2


   sy=sy/sT
   sx=sx/sT
   R2=bgmm'*(sxy-sx*sy)/sqrt(bgmm'*(sxx-sx*sx')*bgmm*(syy-sy^2))
   R2=R2^2
 
   m=zy-zx*bgmm
   J=m'*zwz*m/sige
   df=rows(zx)-k-p
   Jpval=1-cdfc(J,df)  
 
   betastr=string("beta[%2.0f ]= %15.4f", 1:k, bgmm[1:k])	
   betastr=betastr|string("Lag [%2.0f ]= %15.4f", 1:p, bgmm[k+1:k+p])	
   betasestr=string("     %8.4f",SEbgmm)
   tvalues=bgmm./SEbgmm 
   tvalstr=string("  %12.3f", tvalues)    
   DFstr  =      string("N*T=%7.0f",NT)
   DFstr  =DFstr+string("      N=%5.0f",N)
   DFstr  =DFstr+string("      DF=%6.0f              ",df)
   R2str  =string("R-square (differenced model):    %5.4f              ",R2)  
   Jstr   =string("Hansen's J-statistic (p-val):%10.4f ",J)   
   Jstr   =Jstr+string("(%5.4f)     ",Jpval)
   METHstr=string("Instruments according to method: %1.0f                   ",IVmeth)

//TEXTOUTPUT	  

  line1="2-Stage-GMM for the Dynamic Fixed-Effect Model:     "
  line1a="y(i,t) = y(i,t-1)'gamma + x(i,t)'beta + a(i) + e(i,t)"
	line2="=====================================================" 
	line3="PARAMETERS        Estimate        SE          t-value"    

	parametertext=betastr+betasestr+tvalstr 
	output=line1|line1a|line2|line3|line2|parametertext|line2
  output=line2|output|DFstr|R2str|Jstr|METHstr|line2
  
endp  
 

