proc() = plotsplom (x, option, col, xn, xncol)
; -----------------------------------------------------------------------
; Library      plot
; -----------------------------------------------------------------------
; See_also     plot2 plotandrews plotpcp plotgt
; -----------------------------------------------------------------------
; Macro        plotsplom
; -----------------------------------------------------------------------
; Description  Plots a scatterplot matrix from a multivariate dataset x.
; -----------------------------------------------------------------------
; Usage        plotsplom (x {, option {, col {, xn {, xncol}}}})
; Input
;   Parameter  x
;   Definition n x p matrix
;   Parameter  option
;   Definition scalar
;   Parameter  col
;   Definition color
;   Parameter  xn
;   Definition m x p matrix
;   Parameter  xncol
;   Definition color
; -----------------------------------------------------------------------
; Notes        We see in the lower left part of the display matrix the
;              of each variable against any other variable of. 
;              The upper right part
;              will contain the same plots for xn. 
;              The diagonal will contain 1-dimensional
;              plots of x. If you want to plot more than 10 variables
;              you may choose a subset of variables since otherwise
;              you will not see much.     
;
;              option[1,1] allows some transformation on x:
;              grc.prep.inter     -1     interactive selection
;              grc.prep.none       0     no transformation    (default)
;              grc.prep.standard   1     standardize
;              grc.prep.zeroone    2     to [0,1]
;              grc.prep.pcacov     3     pca on cov matrix    
;              grc.prep.pcacorr    4     pca on corr matrix
;              grc.prep.sphere     5     sphering
;
;              option[2,1] decides about the plot on the diagonal:
;              grc.prep.inter     -1     interactive selection
;              grc.splom.none      0     empty
;              grc.splom.boxplotx  1     boxplot x             
;              grc.splom.boxploty  2     boxplot y            (default)
;              grc.splom.dotplotx  3     dotplot x
;              grc.splom.dotplotx  4     dotplot y
;              grc.splom.qqnplot   5     qqplot with normal distribution
; -----------------------------------------------------------------------
; Example      ; loads the library plot
;              library ("plot")
;              ; reads the swiss banknote data (200x6)
;              x = read("bank2")
;              ; computes the principal components of x as scatterplot matrix
;              plotsplom (x, grc.prep.pcacov)
; -----------------------------------------------------------------------
; Result       shows the principal components in a scatterplotmatrix
; -----------------------------------------------------------------------
; Link         ../tutorials/graphicstart.html Introduction to the graphic library
; -----------------------------------------------------------------------
; Keywords    high level graphics, scatterplot matrix
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 970418
; -----------------------------------------------------------------------
  error (cols(x).<2, "plotsplom : cols(x)<2")
  if (cols(x).>10)
    selhead = "Select variables"
    if (exist("x.names")<>1)
      selitem = string("X%.0f", 1:cols(x))
    else
      setitem = x.names
    endif
    sel = selectitem (selhead, selitem)
    if (sum(sel))
      index = paf(1:cols(x), sel)
      x = x[,index]
    endif
  endif
  grc = getglobal ("grc")  
  if (exist("col").<>1)
    col = grc.col.black
  endif
  if (exist("option").<>1)
    option = grc.prep.none
  endif
  if (rows(option)<2)  
    option = option|grc.splom.boxploty
  endif
  grdispsize = cols(x)
  p  = cols(x)
  pn = 0
  if (exist("xn").=1)
    error (cols(xn).<2, "plotsplom : cols(xn)<2")
    error (rows(xn).<>rows(x), "plotsplom : rows(xn)<>rows(x)")
    pn = cols(xn)
    grdispsize = max(grdispsize|pn)
    if (exist("xncol")<>1)
      xncol = grc.col.black
    endif
  endif
  di = pn-p
  if (di.<0)
    di = 0
  endif
  y = transform (x, option[1,1])
  axesoff()
  grdisp = createdisplay(grdispsize, grdispsize)
  {gcol, gind} = groupcol (col, rows(y))
  ng = rows(gcol)
  plotdiag = option[2,1]
  if (plotdiag==grc.splom.inter)
    selhead = "Plot diagonal"
    selitem = "Boxplot X"|"Boxplot Y"|"Dotplot X"|"Dotplot Y"|"QQ Plot"
    do
      sel = selectitem(selhead, selitem)
    until (sum(sel).<2)    
    sel = sum(cumsum(sel))
    if (sel)
      sel = grc.splom.qqnplot+1-sel
    endif
    plotdiag = sel
  endif
; now start drawing
  usedplot = 0~0
  i = 0
  while (i<p)
    i=i+1
    j=0
    while (j<i-1)
      j=j+1
      yij = y[,j]~y[,i]
      setmaskp (yij, col, 3, 3)
      show (grdisp, di+i, j, yij)
      setgopt (grdisp, di+i, j, "ylim", min(y[,i])|max(y[,i]), "xlim", min(y[,j])|max(y[,j]))
      usedplot = usedplot|((di+i)~j)
    endo
; start diagonal
    switch 
    case (plotdiag==grc.splom.boxploty)
      j=0
      while (j.<ng)
        j=j+1
        yij = paf(y[,i], gind.=j)
        yij = grbox(yij, gcol[j])
        yij = grrot(yij, 1)
        yij = grmove(yij, #(1.5*j,0))
        if (j==1)
          show (grdisp, di+i, i, yij)
        else
          adddata (grdisp, di+i, i, yij)
        endif
      endo
      setgopt (grdisp, di+i, i, "ylim", min(y[,i])|max(y[,i]), "xlim", 1.5|(1.5+1.5*ng))
      break
    case (plotdiag==grc.splom.boxplotx)
      j=0
      while (j.<ng)
        j=j+1
        yij = paf(y[,i], gind.=j)
        yij = grbox(yij, gcol[j])
        yij = grmove(yij, #(0,1.5*j))
        if (j==1)
          show (grdisp, di+i, i, yij)
        else
          adddata (grdisp, di+i, i, yij)
        endif
      endo
      setgopt (grdisp, di+i, i, "xlim", min(y[,i])|max(y[,i]), "ylim", 1.5|(1.5+1.5*ng))
      break
    case (option[2,1]==grc.splom.dotploty)
      j=0
      while (j.<ng)
        j=j+1
        yij = paf(y[,i], gind.=j)
        yij = grdot(yij, gcol[j])
        yij = grrot(yij, 1)
        yij = grmove(yij, #(1.5*j,0))
        if (j==1)
          show (grdisp, di+i, i, yij)
        else
          adddata (grdisp, di+i, i, yij)
        endif
      endo
      setgopt (grdisp, di+i, i, "ylim", min(y[,i])|max(y[,i]), "xlim", 1.5|(1.5+1.5*ng))
      break
    case (plotdiag==grc.splom.dotplotx)
      j=0
      while (j.<ng)
        j=j+1
        yij = paf(y[,i], gind.=j)
        yij = grdot(yij, gcol[j])
        yij = grmove(yij, #(0, 1.5*j))
        if (j==1)
          show (grdisp, di+i, i, yij)
        else
          adddata (grdisp, di+i, i, yij)
        endif
      endo
      setgopt (grdisp, di+i, i, "xlim", min(y[,i])|max(y[,i]), "ylim", 1.5|(1.5+1.5*ng))
      break
    case (plotdiag==grc.splom.qqnplot)
      yij = grqqn(y[,i], col)
      show (grdisp, di+i, i, yij)
      setgopt (grdisp, di+i, i, "ylim", min(y[,i])|max(y[,i]), "xlim", min(y[,i])|max(y[,i]))
      break
    endsw
  endo
  if (exist("xn")==1)
    dj = grdispsize-cols(xn)
    i=0
    while (i<cols(xn))
      i=i+1
      j=i
      while (j<cols(xn))
        j=j+1
        yij = xn[,i]~xn[,j]
        setmaskp (yij, xncol, 3, 3)
        show (grdisp, i, dj+j, yij)
        usedplot = usedplot|(i~(dj+j))
      endo
    endo
  endif
; start linking
  i = 1
  rup = rows(usedplot)
  while (i<rup)
    i = i+1
    j = i
    while (j<rup)
      j = j+1
      link (grdisp, usedplot[i,1], usedplot[i,2], grdisp, usedplot[j,1], usedplot[j,2])
    endo
    if (getenv("os")=="unix")
      setmode (grdisp, usedplot[i,1], usedplot[i,2], 1)
    endif
  endo
endp
