proc(rel)=relation(x, colname, opt)
; -----------------------------------------------------------------------
; Library      stats 
; -----------------------------------------------------------------------
; See_also     relationcorr relationchi2 relationcont relationcorrcont
;              relationrank
; -----------------------------------------------------------------------
; Macro        relation
; -----------------------------------------------------------------------
; Description  Computes the relation coefficients (chi^2, contingency,
;              corrected contingency, spearman rank, bravais-pearson)
;              for the data x.
; ----------------------------------------------------------------------
;   Notes      To compute the correlation coefficient 
;              non-interactively you need to set opt = "automatic"|reltype
;              with reltype either "Chi^2", "Contingency", 
;              "Corr. Contigency", "Rank", "Bravais-Pearson" (default).
;
;              In the interactive mode you have to choose one of the 
;              coefficients. Then 
;              you will get a menu sorted after the largest coefficients.
;              If you click on the coefficient you will get some more 
;              information to the corresponding variables.
;
; -----------------------------------------------------------------------
; Usage        rel = relation (x {, colname {, opt}})
; Input
;   Parameter  x 
;   Definition n x p      variables
;   Parameter  colname
;   Definition p x 1      text vector of names of x variables
;   Parameter  opt 
;   Definition q x 1      text vector of optional parameters
; Output
;   Parameter  rel.r      
;   Definition p x p      matrix of correlation coefficients
;   Parameter  rel.pval   
;   Definition p x p      pvalues of an appropriate statistic
; -----------------------------------------------------------------------
; Example      ; loads the library stats
;              library("stats")   
;              ; read swiss banknote data
;              x = read ("bank2")
;              ; compute the correlation coefficients
;              colname = string ("X%0.f", 1:cols(x))
;              relationcorr (x, colname, "automatic"|"Bravais-Pearson")
; -----------------------------------------------------------------------
; Result   Contents of rel.r;
;          [1,]        1  0.23129  0.15176  -0.1898 -0.061321   0.1943 
;          [2,]  0.23129        1  0.74326  0.41378  0.36235 -0.50323 
;          [3,]  0.15176  0.74326        1  0.48676  0.40067 -0.51648 
;          [4,]  -0.1898  0.41378  0.48676        1  0.14185 -0.62298 
;          [5,] -0.061321  0.36235  0.40067  0.14185        1 -0.59404 
;          [6,]   0.1943 -0.50323 -0.51648 -0.62298 -0.59404        1 
;
;          Contents of rel.pval;
;          [1,]        0  0.00098331  0.031935  0.0071053  0.38834  0.0058347 
;          [2,]  0.00098331        0        0  1.1275e-09  1.3454e-07  3.0864e-14 
;          [3,]  0.031935        0        0  2.7089e-13  4.1282e-09  4.885e-15 
;          [4,]  0.0071053  1.1275e-09  2.7089e-13        0  0.045109        0 
;          [5,]  0.38834  1.3454e-07  4.1282e-09  0.045109        0        0 
;          [6,]  0.0058347  3.0864e-14  4.885e-15        0        0        0 
; -----------------------------------------------------------------------
; Author       Sigbert Klinke 970820
; ----------------------------------------------------------------------
  optgiven  = (exist("opt")==2)
  automatic = 0
  colnamegiven = (exist("colname")==2) 
  if (colnamegiven)
    colnamegiven = (rows(colname)==cols(x))
  endif
  if (!colnamegiven)
    colname = string ("X%.0f", 1:cols(x))
  endif

  if (optgiven)
    automatic = sum(opt=="automatic")
  endif
  selitem = "Chi^2"|"Contingency"|"Corr. Contigency"|"Rank"|"Bravais-Pearson"
  if (automatic)
    i = 0
    sel = 0
    while ((i<rows(selitem)) && (sel<1))
      i   = i+1
      sel = sum((selitem[i]==opt).*(1:rows(selitem)))
    endo
    switch
      case(sel==1)
        sel = #(1,0,0,0,0)
        break
      case(sel==2)
        sel = #(0,1,0,0,0)
        break
      case(sel==3)
        sel = #(0,0,1,0,0)
        break
      case(sel==4)
        sel = #(0,0,0,1,0)
        break
      default
        sel = #(0,0,0,0,1)
    endsw     
  else
    selhead = "Coefficient"
    sel = selectitem (selhead, selitem, "single")
  endif
  if (sel[1])
    if (optgiven)
      rel = relationchi2(x, colname, opt)
    else
      rel = relationchi2(x, colname)
    endif
  endif
  if (sel[2])
    if (optgiven)
      rel = relationcont(x, colname, opt)
    else
      rel = relationcont(x, colname)
    endif
  endif
  if (sel[3])
    if (optgiven)
      rel = relationcorrcont(x, colname, opt)
    else
      rel = relationcorrcont(x, colname)
    endif
  endif
  if (sel[4])
    if (optgiven)
      rel = relationrank(x, colname, opt)
    else
      rel = relationrank(x, colname)
    endif
  endif
  if (sel[5])
    if (optgiven)
      rel = relationcorr(x, colname, opt)
    else
      rel = relationcorr(x, colname)
    endif
  endif
endp
