proc(chi) = rrstest(x0,x1,y,score)
; ----------------------------------------------------------------------------
; Library       metrics
; ----------------------------------------------------------------------------
;   Macro       rrstest
; ----------------------------------------------------------------------------
;   Description   Computes the regression rankscore test of a linear
;                 hypothesis based on the dual quantile regression process. 
;                 It tests the hypothesis that b1 = 0 in the quantile
;                 regression model y = x0'b0 + x1'b1 + u.
;                 Test statistic is asymptotically Chi-squared with rank(x1) 
;                 degrees of freedom.
; ----------------------------------------------------------------------------
;   Keywords    Quantile regression
; ----------------------------------------------------------------------------
;   Usage         chi = rrstest(x0,x1,y{,score})
;
;   Input        
;     Parameter   x0  
;     Definition        Matrix of maintained regressors. If there is an intercept term
;                       in the regression, x0 should contain a column of ones.
;     Parameter   x1
;     Definition        Matrix of regressors under test.
;     Parameter   y
;     Definition        Response variable.
;     Parameter   score
;     Definition        The desired score function for test. Default value is 
;                       score = 1 (Wilcoxon).
;                       Currently implemented score functions are Wilcoxon (score = 1), 
;                       Normal (score = 2), and Sign (score = 3) which are
;                       asymptotically optimal for the logistic, Gaussian 
;                       and Laplace error models respectively. Values from
;                       (0,1) are possible too. In this case, sign scores 
;                       generalized to the quantile given by the value of 
;                       score are computed.
;
;   Output
;     Parameter   chi   
;     Definition        Test statistic is asymptotically Chi-squared with 
;                       cols(x1) degrees of freedom.
;//     Parameter   rank  currently not returned
;//     Definition        vector of ranks
; ----------------------------------------------------------------------------
; Example    library("metrics")
;            ; 
;            ; simulate data
;            ;
;            randomize(101)
;            x = uniform(100,3)
;            y = x[,1] + 2*x[,2] - x[,3] + normal(100)
;            ;
;            ; test the hypothesis the coef of x[,3] is zero
;            ;
;            rrstest(x[,1:2], x[,3], y)
;            ;
;            ; once more for a different model
;            ;
;            y = x[,1] + 2*x[,2] + normal(100)
;            ;
;            ; test the hypothesis the coef of x[,3] is zero
;            ;
;            rrstest(x[,1:2], x[,3], y)
; ----------------------------------------------------------------------------
;   Result   Contents of chi for model y = b' * x, b = #(1,2,-1)
;            [1,]   21.902  
;            Contents of chi for model y = b' * x, b = #(1,2,0)
;            [1,]  0.14374 
;            While in the first case has the test statistics high value 
;            (x[,3] is significant in the regression), in the latter case,
;            the test statistics is small (x[,3] is not significant).
; ----------------------------------------------------------------------------
;   See_also rqfit
; ----------------------------------------------------------------------------
;   Author   Pavel Cizek, 990920
; ----------------------------------------------------------------------------
;
; original S form
; check parameter values
  error(exist(x0) <> 1, "Missing data matrix x0!")
  error(exist(x1) <> 1, "Missing data matrix x1!")
  error(exist(y) <> 1, "Missing dependent variable y!")
  z = rqfit(x0, y, -1, 0)
  if (exist(score) <> 1)
    score = 1
  endif
  if ((score <= 0) || (score >= 1))
    if ((score != 1) && (score != 2) && (score != 3))
      score = 1
    endif 
  endif
;
; compute ranks and test statistics
  r = ranks(z, score)
  x1hat = x1 - (x0 * inv(x0'*x0) * x0') * x1     ; qr.resid(qr(x0), x1)
  sn = trans(x1hat) * r.ranks
  chi = (sn' * inv(x1hat' * x1hat) * sn) / r.A2
  rank = r.ranks
endp



;
proc(ranks,A2) = ranks(z,score)
; Function to compute ranks from the  dual  (regression  rankscore)
; process
; USAGE: ranks(z, score)
;     z: regression quantile structure for the model of interest
; score: The  score  function  desired. Currently implemented
;        score functions are Wilcoxon (1), Normal (2), Sign (3) which 
;        are asymptotically optimal for the logistic, Gaussian and
;        Laplace error models respectively. If score lies inside
;        interval (0,1), sign scores generalized to the quantile
;        given by the value of score are computed.
; 
; VALUE:
;       The function returns two components one is the ranks,  the
;       other is a scale factor which is the L_2 norm of the score
;       function.  All score functions  should  be  normalized  to
;       have mean zero.
;
; REFERENCES:
;       Gutenbrunner, C., J. Jureckova,  Koenker, R. and  Portnoy,
;       S.(1993)  Tests  of Linear Hypotheses  based on Regression
;       Rank Scores", Journal of  Nonparametric  Statistics,  (2),
;       307-331.
;
; check missing values
  error(exist(z) <> 9, "Missing rq structure z!")
  error(exist(score) <> 1, "Missing score type!")
  error( ((score <= 0) || (score >= 1)) && (score != 1) && (score != 2) && (score != 3), "Wrong score value!")
;
  A2 = 1
  if (score == 1)
    J = cols(z.sol)
    dt = z.sol[1, 2:J] - z.sol[1, 1:(J-1)]
    ranks = (0.5 * (z.dsol[, 2:J] + z.dsol[, 1:(J-1)]) * trans(dt)) - 0.5
    A2 = 1/12
  else
    if (score == 2) 
      J = cols(z.sol)
      dt = z.sol[1, 2:J] - z.sol[1, 1:(J-1)]
;  !!!!!!!!!!! check what phi is doing... current guess = normal pdf !!!!!!!!!!!!!
      dphi = 0~pdfn(qfn(z.sol[1, 2:(J-1)]))~0
      dphi = diff(trans(dphi))
      ranks = (z.dsol[, 2:J] - z.dsol[, 1:(J-1)]) * (dphi/dt)
    else
      if (score == 3)
        j5 = sum(z.sol[1,]<0.5, 2)
        w = (0.5 - z.sol[1,j5]) / (z.sol[1,j5+1] - z.sol[1,j5])
        r = w * z.dsol[, j5+1] + (1-w) * z.dsol[, j5]
        ranks = 2 * r - 1
      else
        if ((0 < score) && (score < 1))
          isol = sum(z.sol[1,]<score, 2)
          w = (score - z.sol[1,isol]) / (z.sol[1, isol+1] - z.sol[1, isol])
          r = w * z.dsol[, isol+1] + (1-w) * z.dsol[, isol]
          ranks = r - (1 - score)
          A2 = score * (1 - score)
        endif
      endif
    endif
  endif
endp



