#include <stdlib.h>
#include <math.h>

/* Compile into shared library: gcc -shared -O2 -o gplm.so gplm.c */
/*                          or: gcc -G -O2 -o gplm.so gplm.c      */

/* For Borland C++ : Sigbert Klinke 980803                        */

/*******************************************************************/
/**  C-Routines for GPLM  ------  Author: Marlene Mueller 970911  **/
/**                                                               **/
/**  used for exact profile likelihood estimates                  **/
/**                                                               **/
/**  To be used from XploRe 4 via the dlcall command. Can be      **/
/**  also used with Gauss. !! Gauss stores matrices linewise !!   **/
/**  Hence for Gauss x',t' have to be used instead of x. All      **/
/**  other input parameters are vectors, this doesn't matter.     **/
/**                                                               **/
/**  NOTE:                                                        **/
/**                                                               **/
/**   - for each distribution of Y and each link function G we    **/
/**     have to provide the first and second derivative of the    **/
/**     log-likelihood. (for a fisher scoring type procedure,     **/
/**     one has to provide the expected second derivative.)       **/
/**                                                               **/
/**   - <code> stands for the short code of the model as e.g.     **/
/**     'bilo' for binomial with logistic link.                   **/
/**     in the following we use lld<code> and lld<code>fs for the **/
/**     functions that provide the derivatives of the log-lik:    **/
/**       lld<code> is used in Newton-Raphson optimization,       **/
/**       lld<code>fs in Fisher scoring procedures.               **/
/**                                                               **/
/**   - lld<code> and lld<code>fs have 3 inputs:                  **/
/**       nu  -- number of derivative of log-lik.                 **/
/**       eta -- index value at which to compute log-lik.         **/
/**       y   -- response value at which to compute log-lik.      **/
/**                                                               **/
/**  The functions gplmeta<code>, gplmxtilde<code> and eventually **/
/**  gplmeta<code>fs, gplmxtilde<code>fs are the only functions   **/
/**  to be used from outside. They need to get the following      **/
/**  standardized input.                                          **/
/*******************************************************************/
/**
 --------------------------------------------------------------------
     gplmeta<code>, gplmeta<code>fs
 --------------------------------------------------------------------
   Input :
     dim   3 x 1       dimensions n, q, m
     xb    n x 1       x*b
     t     n x q       sorted after 1 col ! normalized by bandwidth !
     y     n x 1       responses
     wx    n x 1       prior weights
     tg    m x q       grid, sorted after 1 col ! normalized by bandwidth !
     etag0 m x 1       initial values for eta on grid
   Output:
     etagh m x 1       updated eta on grid

 --------------------------------------------------------------------
     gplmxtilde<code>, gplmxtilde<code>fs
 --------------------------------------------------------------------
   Input:
     dim    3 x 1       dimensions n, q, p
     xb     n x 1       x*b
     t      n x q       sorted after 1 col ! normalized by bandwidth !
     y      n x 1       responses
     wx     n x 1       prior weights
     eta    n x 1       updated eta
     x      n x p       design x
   Output:
     etah   n x 1       updated eta
     xtilde n x p       updated x
                                                                  **/
/*******************************************************************/

int gplmeta(double *dim,   double    *xb, double *t,
	    double *y,     double    *wx, double *tg,
	    double *etag0, double *etagh, double (*lldfunc)(int,double,double)); // SK 980803
int gplmxtilde(double *dim, double *xb,     double *t,
	       double *y,   double *wx,     double *eta0,
	       double *x,   double *etah,   double *xtilde,
	       double *dnomh, double (*lldfunc)(int,double,double));             // SK 980803

double lldbilo(int nu, double eta, double y)
{ double e, ee, ll;
  e=exp(eta);
  if (nu==1)
    { ll = y - e/(1+e); }         /* 1st derivative of loglikelihood */
  else
    { ee=1+e; ll = -e/(ee*ee); }  /* 2nd derivative of loglikelihood */
  return (ll); }


extern "C" int __export gplmetabilo(double *dim,   double *xb, double *t,
                double *y,     double *wx, double *tg,
                double *etag0, double *etagh)
{ int tmp;
  tmp=gplmeta(dim, xb, t, y, wx, tg, etag0, etagh, lldbilo);
  return (0); }

extern "C" int __export gplmxtildebilo(double *dim, double *xb,   double *t,
		   double *y,   double *wx,   double *eta0,
		   double *x,   double *etah, double *xtilde, double *dnomh)
{ int tmp;
  tmp=gplmxtilde(dim, xb, t, y, wx, eta0, x, etah, xtilde, dnomh, lldbilo);
  return (0); }

double i_cdfn(double x)
{
  const double
    d1 = 0.0498673470, d2 = 0.0211410061, d3 = 0.0032776263,
    d4 = 0.0000380036, d5 = 0.0000488906, d6 = 0.0000053830;
  double u;
  u = fabs (x);
  u = (((((d6*u + d5)*u + d4)*u + d3)*u + d2)*u + d1)*u + 1;
  u = u*u;
  u = u*u;
  u = u*u;
  u = u*u;
  return  x > 0 ? 1 - 0.5/u : 0.5/u;
}

double i_pdfn(double x)
{
  const double pi = 3.141592653589793; double fac;

  fac = 1.0 / sqrt(2.0*pi);
  x = fabs(x);
  return  fac*exp(-0.5*x*x);
}

double lldbipro(int nu, double eta, double y)
{
  double g, gg, g1, g2, g11, ll, feta;

  g =i_cdfn(eta);  gg=g*i_cdfn(-eta);  g1=i_pdfn(eta);
  feta=fabs(eta);

  if (nu==1)
    {
      if (feta>8)
      if (feta>8)
	gg=gg+exp(-0.55*feta);
      ll = (y-g)*g1/(gg);
    }                              /* 1st derivative of loglikelihood */
  else
    {
      if (fabs(y-g)<1.0E-15)
	ll=0;
      else
	{
	  g2 = -eta*g1; g11=g1*g1;

          if (eta>8)
	    ll = 1;
          if (eta<-8);
            ll = -1;
	  if (feta<=8);
	    {
	      g2 = -eta*g1; g11=g1*g1;
	      ll = ( g2/gg - (1-2*g)*g11/(gg*gg) )*(y-g) - g11/gg ;
	    }

	}                          /* 2nd derivative of loglikelihood */
    }
  return (ll); }

double lldbiprofs(int nu, double eta, double y)
{
  double g, gg, g1, ll, feta;

  g =i_cdfn(eta);  gg=g*i_cdfn(-eta);  g1=i_pdfn(eta);
  feta=fabs(eta);

  if (nu==1)
    {
      if (feta>8)
	gg=gg+exp(-0.55*feta);
      ll = (y-g)*g1/(gg);
    }                        /* 1st derivative of loglikelihood */
  else
    {
      if (feta>8)
        ll=0;
      else
	ll = -g1*g1/(gg);
    }                        /* Exp. 2nd derivative of loglikelihood */
  return (ll); }

extern "C" int __export gplmetabipro(double *dim,   double *xb, double *t,
		 double *y,     double *wx, double *tg,
		 double *etag0, double *etagh)
{ int tmp;
  tmp=gplmeta(dim, xb, t, y, wx, tg, etag0, etagh, lldbipro);
  return (0); }

extern "C" int __export gplmetabiprofs(double *dim,   double *xb, double *t,
		   double *y,     double *wx, double *tg,
		   double *etag0, double *etagh)
{ int tmp;
  tmp=gplmeta(dim, xb, t, y, wx, tg, etag0, etagh, lldbiprofs);
  return (0); }

extern "C" int __export gplmxtildebipro(double *dim, double *xb,   double *t,
		      double *y, double *wx,   double *eta0,
		      double *x, double *etah, double *xtilde, double *dnomh)
{ int tmp;
  tmp=gplmxtilde(dim, xb, t, y, wx, eta0, x, etah, xtilde, dnomh, lldbipro);
  return (0); }

extern "C" int __export gplmxtildebiprofs(double *dim, double *xb,   double *t,
		      double *y,   double *wx,   double *eta0,
		      double *x,   double *etah, double *xtilde, double *dnomh)
{ int tmp;
  tmp=gplmxtilde(dim, xb, t, y, wx, eta0, x, etah, xtilde, dnomh, lldbiprofs);
  return (0); }

/*******************************************************************/
/**  General C-Routines for GPLM                                  **/
/*******************************************************************/

double pqua(long p, double *k)
{
  long i;
  double q = 1.0, r;
  for (i=0; i<p; i++)
  {
    r = k[i]*k[i];
    if (r>1.0) return 0.0;
    r = 1.0-r;
    r = 0.9375*r*r;
    q *= r;  /* q = q * r */
  }

  return (q);
}

int gplmeta(double *dim,   double    *xb, double *t,
	    double *y,     double    *wx, double *tg,
	    double *etag0, double *etagh, double (*lldfunc)(int,double,double))

{
  long   i, j, k, l, istart=0, nn, qq, mm;
  double nom, denom, *kern, w, e, g, ll1, ll2;

  nn=*(dim+0); qq=*(dim+1); mm=*(dim+2);

  kern  = (double *) malloc(sizeof(double)*qq);

  for (j=0; j<mm; j++)               /* loop over grid */
  {
    denom = 0; nom=0;

    for (i=istart; i<nn; i++)               /* loop over obs  */
      {
	kern[0] = *(t+i) - *(tg+j);

	if (kern[0]<-1.0)
	  istart=i+1;
	else
	  {
	    if (kern[0]>1.0) break;
	    for (k=1; k<qq; k++)
	      kern[k] = *(t+i+k*nn) - *(tg+j+k*mm);

	    w = pqua(qq, kern);
	    w = *(wx+i) *w;

	    ll1=lldfunc(1, *(xb+i)+ *(etag0+j), *(y+i));  /* ll'  */
	    ll2=lldfunc(2, *(xb+i)+ *(etag0+j), *(y+i));  /* ll'' */

	    nom   = nom   + w*ll1;
	    denom = denom + w*ll2;

	  }
      }
    if (denom!=0)
      *(etagh+j)= *(etag0+j) - nom/denom;
    if (denom==0)
      {
	if (nom==0)
	  *(etagh+j)= *(etag0+j);
	else
	  *(etagh+j)= *(etag0+j) - (nom+1.0E-16)/(denom+1.0E-16);
      }
  }

  free (kern);

  return (0);
}

int gplmxtilde(double *dim, double *xb,     double *t,
	       double *y,   double *wx,     double *eta0,
	       double *x,   double *etah,   double *xtilde,
	       double *dnomh, double (*lldfunc)(int,double,double))
{
  long   i, j, k, l, istart, nn, qq, pp;
  double nom, *nomx, denom, *kern, *kernw, w, e, g, ll1, ll2;

  nn=*(dim+0); qq=*(dim+1); pp=*(dim+2);

  kern  = (double *) malloc(sizeof(double)*qq);
  nomx  = (double *) malloc(sizeof(double)*pp);

  for (istart=j=0; j<nn; j++)         /* start loop over j */
    {

      nom=0; denom=0;                 /* initializations */
      for (l=0; l<pp; l++)
	nomx[l]=0;                    /* l: loop over dim(x) = p */

      for (i=istart; i<nn; i++)       /* start loop over i */
	{
	  kern[0] = *(t+i) - *(t+j);

	  if (kern[0]<-1.0)
	    istart=i+1;
	  else
	    {
	      if (kern[0]>1.0) break;
	      for (k=1; k<qq; k++)
		kern[k] = *(t+i+k*nn) - *(t+j+k*nn);

	      w = pqua(qq, kern);
	      w = *(wx+i) *w;

	      ll1=lldfunc(1, *(xb+i)+ *(eta0+j), *(y+i));  /* ll'  */
	      ll2=lldfunc(2, *(xb+i)+ *(eta0+j), *(y+i));  /* ll'' */

	      nom  = nom  + w*ll1;

	      w=w*ll2;
	      denom = denom + w;

	      for (l=0; l<pp; l++)
		nomx[l] = nomx[l] + w* *(x+i+l*nn);
	    }
	}                             /* i: end */

      *(dnomh+j) = denom;

      if (denom!=0)
	{
	  *(etah+j)= *(eta0+j) - nom/denom;
	  for (l=0; l<pp; l++)
	    *(xtilde+j+l*nn)= *(x+j+l*nn) - nomx[l]/denom;
	}

      if (denom==0)
	{
	  if (nom==0)
	    {
	      *(etah+j)= *(eta0+j);
	      for (l=0; l<pp; l++)
		*(xtilde+j+l*nn)= *(x+j+l*nn);
	    }
	  else
	    {
	      denom=denom+1.0E-16;
	      *(etah+j)= *(eta0+j) - (nom+1.0E-16)/denom;
	      for (l=0; l<pp; l++)
		*(xtilde+j+l*nn)= *(x+j+l*nn)-(nomx[l]+1.0E-16)/denom;
	    }
	}
    }                                 /* j: end */

  free (kern);
  free (nomx);

  return (0);
}


