proc(A,b,V,ctrl) = ICerzsep(e,S1,S2,A0,b0,N,eps,itmax,expl,fact0,aus)

; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also      rICfil,calibrIC
; -----------------------------------------------------------------
;   Macro        ICerzsep
; -----------------------------------------------------------------
;   Description  Auxiliary routine for rICfil:
;
;                - if possible - generates for  
;                Lambda=Lambda1+Lambda2, 
;                Lambda1~N(0,S1), Lambda2~N(0,S2) indep
;                a Hampel-Krasker-IC    psi
;                to efficiency loss  e, 
;                i.e. 
;
;                E psi Lambda' = EM,  E psi=0 (1)
;
;                E |psi|^2= (1+e) tr ((S1+S2)^{-1})
; 
;                and psi= A (Lambda1 w_b + Lambda2)
;
;                w_b=min(1,b/|A Lambda1|)
;
;                For dim p==1 a Newton-Algo is used
;                     for both a and b, 
;                for dim p>=2 for A a fixed-point-algorithm 
;                     and for b a "careful" bisection method 
;                      is used. 
;                Integration for A and p==2 is done by 
;                    a Romberg-procedure.  
;                Integration for A and p>2 is done by 
;                    a MC-procedure.
; -----------------------------------------------------------------
;   Keywords     robust Kalman filter, rIC
; -----------------------------------------------------------------
;   Usage          {A,b,V,ctrl}=ICerzsep(e,S1,S2,A0,b0,N,eps,itmax,expl,fact0,aus)
;   Input
;     Parameter   e
;     Definition     numeric; efficiency loss to attain
;     Parameter   S1
;     Definition     p x p; Covariance of the first (clipped) component
;     Parameter   S2
;     Definition     p x p; Covariance of the second (unclipped) component
;     Parameter   A0  
;     Definition     p x p; starting value for A; if 0 is entered I^{-1} is taken
;                                 else if p>1 and dim(A0)==1 FI^{-1}*A0 is taken
;                                  and if p==1 and A0<0  , -A0/FI is taken     
;     Parameter   b0    
;     Definition     numeric; starting value for clipping height; if 0 is entered
;                        4*max(vec(abs("A0"))) [i.e. perhaps modified]
;     Parameter    N
;     Definition      integer; MC-sample size / integration grid-points
;     Parameter    eps
;     Definition      numeric; exactitude
;     Parameter    itmax
;     Definition      integer; maximal number of ICerzseptions
;     Parameter    expl
;     Definition      numeric; threshold for the changes in abs. value of A: 
;                        beyond this value convergence is uncertain  ~ 4
;     Parameter    fact0
;     Definition      numeric (>1)!; factor determining how fast we descend from b to b
;     Parameter    aus
;     Definition      integer; 0: no output during execution, 1: some output, 2: more output, 3: a lot of output
; Output                                                           
;     Parameter   A
;     Definition     p x p;  Lagrange-Multiplyer solving (1)
;     Parameter   b
;     Definition     1;  clipping height
;     Parameter   V
;     Definition     p x p;  corresponding Covariance
;     Parameter   ctrl
;     Definition     integer; tells if convergence "happened"
; -----------------------------------------------------------------
; Notes
;      Not to be used outside of rICfil!
;
;      Is called by calibrIC.
;
;      Calles [within kalman.lib]  iteras, absepnewton.
; 
;      The Output produced if aus>=1 is not identical with the output parameters.
;      (aus>=1) is for interactive use ,  success is controlled by "verbal" output. 
;      (aus==0) is for other quantlets calling ICerz,  
;                       success is controlled by variable "ctrl".
; -----------------------------------------------------------------
;   Example   to be looked up in rICfil
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel 991010 
; -----------------------------------------------------------------

FI=S1+S2  
p=rows(FI)
ctrl=0

if (max(abs(vec(FI)))==0)
    FIs=1
else
    FIs=FI
endif 

Imax=1.6*sqrt(max(vec(abs( ginv(FI)*S1*ginv(FI) ) ) ) )

switch

  case(p==1)

   a4=(aus>3)
 
   error(e>S2/FI,"e=_e_ too large!")
    
    switch
       case (A0<0)
               At=abs(A0)/FI
       break
       
       case (A0==0)
               At=1/FI
       break
       
       case (A0>0)
               At=A0
       break
       
       default
       break
   
    endsw
        
    if (b0<=0)
       bt=Imax
    else
       bt=b0
    endif
    
    At0=At
    bt0=bt
      
    ctrl=1
    while ((ctrl!=0)&&(bt<100*Imax))
       erg=absepnewton(e,S1,S2,itmax,eps,At,bt,a4)
       At=At0*1.2
       bt=bt*1.1
       ctrl=erg.ctrl+(erg.A<0)+(erg.b<0)
       if (aus>1)
           "problems finding b s.t. algo converges... at this stage b=_bt_"
       endif
    endo
    if (bt>=100*Imax)
    bt=bt0
    At=At0
    while ((ctrl!=0)&&(bt>Imax/100))
       erg=absepnewton(e,S1,S2,itmax,eps,At,bt,a4)
       At=At0*1.2
       bt=bt/1.1
       ctrl=erg.ctrl+(erg.A<0)+(erg.b<0)
       if (aus>1)
           "problems finding b s.t. algo converges... at this stage b=_bt_"
       endif
    endo
    endif
    A=erg.A
    b=erg.b
    V=(1+e)/FI

  break


  case (p>1) 
    
    ;Calculation of the trace of FIs^{-1}
    trsit=unit(p).*ginv(FIs)
    trsit=sum(vec(trsit)); trace of classically optimal V

    fact=abs(fact0)
    if (fact <1) 
        fact=1/fact
    endif
     
    if (b0<=0)
        b=Imax
    else
        b=b0
    endif

    if (aus>0)
       "finding a (large) b that works"
    endif

    abr=1

    while(abr!=0)
       
       erg=iteras(A0,S1,S2,b,N,eps,itmax,expl)
       At=erg.A
       abr=erg.ctrl
       trV=sum(vec(unit(p).*erg.V))     ; trace of V
       eff=trV/trsit-1
       abr=abr*(abs(eff-e)>eps); exit the loop, if eff is close to e, too
       
       if (aus>2)
           "first admissible clipping-b  (at this stage) b0=_b_"
           "Lagrange-A(b0) =_At_"
           "rel eff. loss (b0)=_eff_"
       endif
      
       if (abr==1)
           fact=fact^1.5 
       endif
      
       b=b * fact
    
    endo 
      
    b=b/fact
    
    if ((abs(eff-e)<eps)&&(erg.ctrl==0))
        A=At
        V=erg.V
        ctrl=0
        abr=0
    else
        abr=1
    endif    
    
    if ((aus==1)||(aus==2))
           "first admissible clipping-b  b0=_b_"
           "Lagrange-A(b0) =_At_"
           "rel eff. loss (b0)=_eff_"
    endif
    
    i=0
    
    if (abr>0)
        
        if (aus>0)
            "finding two admissible (A,b) - points for a bisection method"
        endif

        if (eff>e)
       

            if (aus>0)
                "easier case: finding a right admissible b-point "
                "with smaller rel.eff. loss than e"
                "and b-left=b0"
            endif
       
            Atl=At
            bl=b
            effl=eff
            Vl=erg.V
         
        
            while ( ((abr!=0)||((eff>e)&&(abs(eff-e)>eps)))&&(b<10*Imax))

               erg=iteras(ginv(FIs),S1,S2,b,N,eps,itmax,expl)

               abr=erg.ctrl
               trV=sum(vec(unit(p).*erg.V))     ; trace of V
               eff=trV/trsit-1

               if (abr>1)
                   b=b/fact^2
               endif
               if (abr!=0)
                   b=b*sqrt(fact)
               endif
               b=b *fact

               if (aus>2)
                   "b-right  (at this stage) b0=_b_"
                   "Lagrange-A(b-right) =_At_"
                   "rel eff. loss (b-right)=_eff_"
               endif
        

            endo 


            Atr=erg.A

            br=b
            Atr=At
            effr=eff
            Vr=erg.V
            
            if ((abs(eff-e)<eps)&&(erg.ctrl==0))
                A=At
                V=erg.V
                ctrl=0
            endif    
 

        else  ; eff<e

            if (aus>0)
                "more difficult case: finding a left admissible b-point" 
                "with bigger eff than e"
                "and b-right=b0"
            endif

            br=b
            Atr=At
            effr=eff
            Vr=erg.V

            bd=1
            noch=1
        
            nA=max(abs(vec(At)))

            while (noch==1)
             
               i=i+1
     
               while (bd<eps)
                  bd=bd*sqrt(fact)
               endo
    
               bd=bd/fact
               b=b-bd
      
              ; mit zulaessigem Start-A bewegen wir uns auf b zu
      
               nA0=nA

               erg=iteras(At,S1,S2,b,N,eps,itmax,expl)
               abr=erg.ctrl
               trV=sum(vec(unit(p).*erg.V))     ; trace of V
               eff=trV/trsit-1

               if (aus>2)
                   "b-left  (at this stage) b0=_b_"
                   "Lagrange-A(b-left) =_At_"
                   "rel eff. loss (b-left)=_eff_"
               endif

               if (erg.ctrl==0)

                    At=erg.A
                    nA=max(abs(vec(At)))
         
                    if (nA/nA0>expl)
                        bd=bd/fact
                    endif
          
                    if (abs(nA/nA0-1)<0.1)
                        bd=bd*fact
                    endif

               endif
           
   
     
               if ((eff>e)&&(abr==0))
                    noch=0
                    Atl=At
                    effl=eff
                    bl=b
                    Vl=erg.V
               endif
      
               if ((abs(eff-e)<eps)&&(abr==0))
                   noch=0
                   A=At
                   V=erg.V
                   ctrl=0
               endif
               
               if (abr!=0)
                   b=b+bd
               endif
          
               error((i>10*itmax), "max. nb. of iter.'s reached; no convergence!")
      
            endo ;searching a left admissible point

      
        endif ; finding left and right endpoint




    
        if ((i<10*itmax)&&(abs(eff-e)>eps))

            db=br-bl
            i=0

            if (aus>0)
                "now a 'careful' bisection method"
                "b-left =_bl_"
                "Lagrange-A(b-left) =_Atl_"
                "rel eff. loss (b-left)=_effl_"
                "Variance (b-left)=_Vl_"
                "b-right =_b_"
                "Lagrange-A(b-right) =_Atr_"
                "rel eff. loss (b-right)=_effr_"
                "Variance (b-right)=_Vr_"
            endif


            while ((db>eps)&&(i<itmax)&&(abs(eff-e)>eps))

               ;bisevtion method

               i=i+1
               b=(bl+br)/2
               br0=b
               abr=1
               At=(Atl+Atr)/2

               ; searching an admissible intermediate point between bl and br
               ; as the algorithm will not necessarily converge in the midpoint

               if (aus>1)
                 "searching an admissible intermediate point between bl and br"
                 "as the algorithm will not necessarily converge in the midpoint"
               endif 

               while ((abr!=0)&&(br-br0>eps)&&(abs(eff-e)>eps))
                  
                  b=br0
                  erg=iteras(At,S1,S2,b,N,eps,itmax,expl)
                  br0=(br+br0)/2 
                  At=(At+Atr)/2
                  abr=erg.ctrl
                  trV=sum(vec(unit(p).*erg.V))     ; trace of V
                  eff=trV/trsit-1
                
                  if (aus>2)
                      "b-mid (at this stage) =_b_"
                      "Lagrange-A(b-mid) =_erg.A_"
                      "rel.eff. loss(b-mid) =_eff_"
                 endif
              
               endo
              
               if (eff>e)
                   bl=b
                   Atl=At
                   effl=eff
                   Vl=erg.V
               else
                   br=b
                   Atr=At
                   effr=eff
                   Vr=erg.V
               endif

               if (aus==2)
                      "b-mid =_b_"
                      "Lagrange-A(b-mid) =_erg.A_"
                      "rel.eff. loss(b-mid) =_eff_"
               endif
            
               db=br-bl   
              
            endo
            ctrl=erg.ctrl
            if ((eff-effr)>(effl-eff))
                b=bl
                A=Atl
                eff=effl
                V=Vl
            else
                b=br
                A=Atr
                eff=effr
                V=Vr
            endif
        else
            if (abs(eff-e)>eps)
                ctrl=abr+ctrl+16
            endif
        endif
    endif
    
    A=At
    V=erg.V

    if (aus>0)
       "clipping-b =_b_"
       "Lagrange-A(b) =_A_"
       "rel eff. loss (b)=_eff_"
       "Variance (b)=_V_"
    endif
    
    break
    
    default
    break 

endsw
endp
  
