proc (res) = SPKRexpcov (r, d, alpha, se)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRsurfgls SPKRgaucov SPKRsphercov
;              SPKRtrmat SPKRprmat SPKRsemat SPKRcorrelogram SPKRvariogram
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRexpcov
; -----------------------------------------------------------------------
; Description  Spatial covariance function for use with SPKRsurfgls.
; -----------------------------------------------------------------------
; Usage        covvals = SPKRexpcov (r, d, alpha, se)
; Input
;   Parameter  r
;   Definition n x 1  -  vector of distances at which to evaluate the 
;                       covariance
;   Parameter  d
;   Definition scalar -  range parameter
;   Parameter  alpha
;   Definition sclar  -  proportion of nugget effect
;   Parameter  se
;   Definition scalar -  standard deviation at distance zero
; Output
;   Parameter  res
;   Definition n x 1  -  vector of covariance values 
; -----------------------------------------------------------------------
; Notes        Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; calculate spatial covariance function SPKRexpcov
;              ; with d = 0.7, alpha = 0, and se = 1 on a grid
;              ; from 0 through 7 
;              ddat = grid(0, 0.1, 71)
;              linedat = ddat ~ SPKRexpcov (ddat, 0.7, 0, 1)
;              ; create a display and draw the covariance
;              disp = createdisplay (1, 1)
;              show (disp, 1, 1, linedat)
; -----------------------------------------------------------------------
; Result       A vector of covariance values. Typically, this quantlet
;              is used together with SPKRsurfgls or SPKRcorrelogram.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("r") != 1, "r must be numeric")
  error (exist("d") != 1, "d must be numeric")
  error (d <= 0, "d must be > 0")
  error (exist("alpha") != 1, "alpha must be numeric")
  error ((alpha < 0) || (alpha > 1), "alpha must be >= 0 and <= 1")
  error (exist("se") != 1, "se must be numeric")
  error (se < 0, "se must be >= 0")
  ;
  ;
  res = se^2 * (alpha * (r < d / 10000) + (1-alpha) * exp(-r/d))
endp
