proc (res) = SPKRmultcontours (disp, pos1, pos2, obj, start, end, step)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRsurfgls SPKRexpcov SPKRgaucov SPKRsphercov 
;              SPKRtrmat SPKRprmat SPKRsemat SPKRcorrelogram SPKRvariogram
; -----------------------------------------------------------------------
; Macro        SPKRmultcontours
; -----------------------------------------------------------------------
; Description  Draws multiple contour lines of a spatial object of type
;              "trmat", "prmat", or "semat".
; -----------------------------------------------------------------------
; Usage        cont = SPKRmultcontours (disp, pos1, pos2, obj, start, end, step)
; Input
;   Parameter  disp
;   Definition display -  name of the display 
;   Parameter  pos1
;   Definition scalar -  row of the display 
;   Parameter  pos2
;   Definition scalar -  column of the display 
;   Parameter  obj
;   Definition list   -  (of type "trmat", "prmat", or "semat") calculated 
;                       by SPKRtrmat, SPKRprmat, or SPKRsemat
;   Parameter  start
;   Definition scalar -  starting value for contour lines 
;   Parameter  end
;   Definition scalar -  ending value for contour lines
;   Parameter  step
;   Definition scalar -  step width between two contour lines
; Output
;   Parameter  res
;   Definition n x 2  -   matrix of all points at all levels of the
;                        contour lines
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;              library ("spatial")
;              ; read a spatial data set
;              topo = read("topo.dat")
;              ; calculate a polynomial regression surface of order 2
;              myres = SPKRsurfls (2, topo)
;              ; calculate a 30 x 30 grid within [0, 6.5] x [0, 6.5]
;              ; using trend surface myres
;              mygrid = SPKRtrmat (myres, 0, 6.5, 0, 6.5, 30)
;              ; create a display and draw contour lines in the range
;              ; 750 through 975, using steps of width 25
;              disp = createdisplay (1, 1)
;              cont = SPKRmultcontours(disp, 1, 1, mygrid, 750, 975, 25)
; -----------------------------------------------------------------------
; Result       A plot of multiple countour lines is drawn and the 
;              corresponding points are being returned.
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("disp") != 4, "disp must be a display")
  error (exist("pos1") != 1, "pos1 must be numeric")
  error (exist("pos2") != 1, "pos2 must be numeric")
  error (exist("obj") != 9, "obj must be a list")
  error (comp(obj, "type") == 0, "obj must contain component type")
  error ((obj.type != "trmat") && (obj.type != "prmat") && (obj.type != "semat"), "obj.type must be trmat, prmat, or semat")
  error (exist("start") != 1, "start must be numeric")
  error (exist("end") != 1, "end must be numeric")
  error (end <= start, "end must be > start")
  error (exist("step") != 1, "step must be numeric")
  error (step <= 0, "step must be > 0")
  ;
  ;
  objsort = sort(obj.z, #(1, 2, 3))
  ;
  ; plot empty frame
  ;
  linedat1 = #(obj.xl, obj.xu) ~ #(obj.yl, obj.yu)
  setmaskp (linedat1, 0, 0, 0)
  ;setmaskl (linedat1, #(1, 2)', 0, 0, 0)
  show (disp, pos1, pos2, linedat1)  
  ;
  ; draw all contours (as lines)
  ;
  i = start
  {lt, pt} = contour2(objsort, i)
  allpt = pt
  ;
  setmaskp (pt, 0, 0, 0)
  setmaskl (pt, lt, 0, 1, 2)
  adddata (disp, pos1, pos2, pt)
  ;
  minx = paf(pt, pt[,1] == min(pt[,1]))
  ptstr = string("%.0f", i)
  j = 2
  while (j <= rows(minx))
    ptstr = ptstr | string("%.0f", i)
    j = j + 1
  endo
  setmaskp (minx, 0, 0, 0)
  setmaskt (minx, ptstr, 1, 0, 12)
  adddata (disp, pos1, pos2, minx)
  ;
  do
    i = i + step
    {lt, pt} = contour2(objsort, i)
    allpt = allpt | pt
    ;
    setmaskp (pt, 0, 0, 0)
    setmaskl (pt, lt, 0, 1, 2)
    adddata (disp, pos1, pos2, pt)
    ;
    minx = paf(pt, pt[,1] == min(pt[,1]))
    ptstr = string("%.0f", i)
    j = 2
    while (j <= rows(minx))
      ptstr = ptstr | string("%.0f", i)
      j = j + 1
    endo
    setmaskp (minx, 0, 0, 0)
    setmaskt (minx, ptstr, 1, 0, 12)
    adddata (disp, pos1, pos2, minx)
    ;
  until (i >= end)
  res = allpt
endp
