proc (res) = SPKRsemat (obj, xl, xu, yl, yu, n, se)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfls SPKRsurfgls SPKRexpcov SPKRgaucov SPKRsphercov 
;              SPKRtrmat SPKRprmat SPKRcorrelogram SPKRvariogram
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRsemat
; -----------------------------------------------------------------------
; Description  Evaluates a Kriging standard error of prediction surface 
;              over a grid.
; -----------------------------------------------------------------------
; Usage        mygrid = SPKRsemat (obj, xl, xu, yl, yu, n {, se})
; Input
;   Parameter  obj
;   Definition list   -  (of type "trgls") calculated by SPKRsurfgls
;   Parameter  xl
;   Definition scalar -  lower limit in x-direction of the rectangle for grid
;   Parameter  xu
;   Definition scalar -  upper limit in x-direction of the rectangle for grid
;   Parameter  yl
;   Definition scalar -  lower limit in y-direction of the rectangle for grid
;   Parameter  yu
;   Definition scalar -  upper limit in y-direction of the rectangle for grid
;   Parameter  n
;   Definition scalar -  use n x n grid within the rectangle
;   Parameter  se
;   Definition scalar -  optional; standard error at distance zero as a 
;                       multiple of the supplied covariance; if omitted then
;                       estimated under the assumption that a correlation
;                       function has been applied before
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       z, xl, xu, yl, yu, n, type --
;   Parameter  z
;   Definition grid   -  for further useage with SPKRmultcontours
;   Parameter  xl
;   Definition scalar -  same as input value xl
;   Parameter  xu
;   Definition scalar -  same as input value xu
;   Parameter  yl
;   Definition scalar -  same as input value yl
;   Parameter  yu
;   Definition scalar -  same as input value yu
;   Parameter  n
;   Definition scalar -  same as input value n
;   Parameter  type
;   Definition string -  "semat"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       topo = read("topo.dat")
;	       ; calculate a trend surface of order 2, using expcov as
;              ; spatial covariance function with dval = 0.7, alph = 0,
;              ; and se = 1
;  	       myres = SPKRsurfgls(2, 0, topo, 1000, 0.7, 0, 1)
;  	       ; calculate a 30 x 30 grid within [0, 6.5] x [0, 6.5]
;  	       ; using trend surface myres
;  	       mygrid = SPKRsemat (myres, 0, 6.5, 0, 6.5, 30)
; -----------------------------------------------------------------------
; Result       A grid suitable for display through SPKRmultcontours.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("obj") != 9, "obj must be a list")
  error (comp(obj, "type") == 0, "obj must contain component type")
  error (obj.type != "trgls", "obj.type must be trgls")
  error (exist("xl") != 1, "xl must be numeric")
  error (exist("xu") != 1, "xu must be numeric")
  error (exist("yl") != 1, "yl must be numeric")
  error (exist("yu") != 1, "yu must be numeric")
  error (exist("n") != 1, "n must be numeric")
  error (n < 1, "n must be >= 1")
  error (n >= 1000, "n must be < 1000")
  ;
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall("_VR_frset", obj.minx[1,1], obj.maxx[1,1], obj.minx[1,2], obj.maxx[1,2])
  else  
    ; Unix
    d = dlcall("VR_frset", obj.minx[1,1], obj.maxx[1,1], obj.minx[1,2], obj.maxx[1,2])
  endif
  ;
  alph = obj.alph
  nalph = rows(alph)
  ;
  if (os == "windows")
    d = dlcall("_VR_alset", alph, nalph)
  else  
    ; Unix
    d = dlcall("VR_alset", alph, nalph)
  endif
  ;
  dx = (xu - xl) / n
  dy = (yu - yl) / n
  x = grid (xl, dx, n+1)
  y = grid (yl, dy, n+1)
  z = grid (#(xl,yl, 0), #(dx, dy, 1), #(n + 1, n + 1, 1))
  np = obj.np
  npar = ((np + 1) * (np + 2)) / 2
  ;
  if (exist("se") == 0) ; parameter se has not been assigned
    se = sqrt(sum(obj.W^2) / (rows(obj.W) - npar))
  endif
  ;
  i = 1
  do
    yrep = grid (y[i], 0, rows(x))
    ;
    npt = rows(x)
    seval = 1:npt
    nobjx = rows(obj.x)
    ;
    if (os == "windows")
      d = dlcall("_VR_prvar", seval, x, yrep, npt, obj.x, obj.y, obj.l, obj.r, nobjx, np, npar, obj.l1f)
    else  
      ; Unix
      d = dlcall("VR_prvar", seval, x, yrep, npt, obj.x, obj.y, obj.l, obj.r, nobjx, np, npar, obj.l1f)
    endif
    ;
    z[(i-1)*rows(x)+1:i*rows(x), 3] = se * sqrt(seval)
    i = i+1
  until (i > rows(y))
  type = "semat"
  res = list(z, xl, xu, yl, yu, n, type)
endp
