proc (res) = SPKRsurfls (np, xmat)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPKRsurfgls SPKRexpcov SPKRgaucov SPKRsphercov 
;              SPKRtrmat SPKRprmat SPKRsemat SPKRcorrelogram SPKRvariogram
;              SPKRmultcontours
; -----------------------------------------------------------------------
; Macro        SPKRsurfls
; -----------------------------------------------------------------------
; Description  Fits a trend surface, i.e., a polynomial regression
;              surface, by least squares.
; -----------------------------------------------------------------------
; Usage        myres = SPKRsurfls (np, xmat)
; Input
;   Parameter  np
;   Definition scalar -  degree of the polynomial surface, an integer 
;                       in the range 0..6
;   Parameter  xmat
;   Definition n x 3  -  matrix of locations (x_i, y_i) [columns 1 & 2] and 
;                       observations z_i [column 3]
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       x, y, z, np, f, r, beta, wz, minx, maxx, type --
;   Parameter  x
;   Definition n x 1  -  same as xmat[,1] 
;   Parameter  y
;   Definition n x 1  -  same as xmat[,2] 
;   Parameter  z
;   Definition n x 1  -  same as xmat[,3] 
;   Parameter  np
;   Definition scalar  - same as input value np
;   Parameter  f
;   Definition matrix  - internal use only
;   Parameter  r
;   Definition matrix  - internal use only
;   Parameter  beta
;   Definition matrix -  (np + 1)(np + 2) / 2 coefficients
;   Parameter  wz
;   Definition matrix -  internal use only
;   Parameter  minx
;   Definition 1 x 3  -  minimum of columns of xmat
;   Parameter  maxx
;   Definition 1 x 3  -  maximum of columns of xmat
;   Parameter  type
;   Definition string  - "trls"
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       topo = read("topo.dat")
;	       ; calculate a polynomial regression surface of order 2
;  	       myres = SPKRsurfls (2, topo)
; -----------------------------------------------------------------------
; Result       A list consisting of input parameters, intermediate
;              results, and final results of a polynomial regression
;              surface. This list will be used in other spatial
;              statistics quantlets such as SPKRtrmat, SPKRcorrelogram,
;              or SPKRvariogram.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("np") != 1, "np must be numeric")
  error ((np < 0) || (np > 6), "np must be >= 0 and <= 6")
  error (exist("xmat") != 1, "xmat must be a matrix")
  error (cols(xmat) < 3, "xmat must have at least 3 columns")
  ;
  ;
  x = xmat[,1]
  y = xmat[,2]
  z = xmat[,3]
  minx = min(xmat)
  maxx = max(xmat)
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall("_VR_frset", minx[1,1], maxx[1,1], minx[1,2], maxx[1,2])
  else  
    ; Unix
    d = dlcall("VR_frset", minx[1,1], maxx[1,1], minx[1,2], maxx[1,2])
  endif
  ;
  n = rows(xmat)
  npar = ((np + 1) * (np + 2)) / 2
  f = 1:(n * npar)
  ;
  if (os == "windows")
    d = dlcall("_VR_fmat", f, x, y, n, np)
  else  
    ; Unix
    d = dlcall("VR_fmat", f, x, y, n, np)
  endif
  ;
  r = 1:((npar * (npar+1)) / 2)
  beta = 1:npar
  wz = 1:n
  ifail = 0
  ;
  if (os == "windows")
    d = dlcall("_VR_ls", x, y, z, n, np, npar, f, r, beta, wz, ifail)
  else  
    ; Unix
    d = dlcall("VR_ls", x, y, z, n, np, npar, f, r, beta, wz, ifail)
  endif
  ;
  error (ifail > 0, "Error in Householder decomposition")
  type = "trls"
  res = list (x, y, z, np, f, r, beta, wz, minx, maxx, type)
endp
