proc (res) = SPPPkenvl (fs, k, obj)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPPPinit SPPPinitrandom SPPPsetregion SPPPgetregion
;              SPPPkfn SPPPpsim SPPPstrauss SPPPssi SPPPkaver
; -----------------------------------------------------------------------
; Macro        SPPPkenvl
; -----------------------------------------------------------------------
; Description  Computes envelope (upper and lower limits) and average 
;              of simulations of K-fns.
; -----------------------------------------------------------------------
; Usage        ppkenvl = SPPPkenvl (fs, k, obj)
; Input
;   Parameter  fs
;   Definition scalar -  full scale of the plot
;   Parameter  k
;   Definition scalar -  number of regularly spaced distances in (0, fs)
;   Parameter  obj
;   Definition list   -  (of type "PSIM", "STRAUSS", or "SSI") consisting 
;                       of xmat, ymat, and type, caculated by SPPPpsim,
;                       SPPPstrauss, or SPPPssi
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       hx, hl, hu, aver --
;   Parameter  hx
;   Definition k x 1  -  vector of distances
;   Parameter  hl
;   Definition k x 1  -  vactor of min of K-fn
;   Parameter  hu
;   Definition k x 1  -  vector of max of K-fn
;   Parameter  aver
;   Definition k x 1  -  vector of averages of K-fn
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       pines = read ("pines.dat")
;	       ; initialize a point process
;  	       pinesobj = SPPPinit (pines, 0, 96, 0, 100, 10)
;  	       ; reset random number generator
;              SPPPinitrandom (0)
;              ; simulate 100 Binomial point processes, each consisting
;              ; of 72 points, in the previously determined domain
;  	       ppsim = SPPPpsim (100, 72)
;              ; calculate the envelope in interval (0, 5), split 
;              ; into 100 segments
;              ppkenvl = SPPPkenvl (5, 100, sppsim)
; -----------------------------------------------------------------------
; Result       List, consisting of distances and min, max, and averages 
;              of K-fn.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("fs") != 1, "fs must be numeric")
  error (fs <= 0, "fs must be > 0")
  error (exist("k") != 1, "k must be numeric")
  error (k < 1, "k must be >= 1")
  error (exist("obj") != 9, "obj must be a list")
  error (comp(obj, "type") == 0, "obj must contain component type")
  error ((obj.type != "PSIM") && (obj.type != "STRAUSS") && (obj.type != "SSI"), "obj.type must be PSIM, STRAUSS, or SSI")
  error (comp(obj, "xmat") == 0, "obj must contain component xmat")
  error (comp(obj, "ymat") == 0, "obj must contain component ymat")
  ;
  ;
  pp = obj.xmat[,1] ~ obj.ymat[,1]
  area = SPPPgetregion ()
  type = "PP"
  ppobj = list (pp, area, type)
  h = SPPPkfn (ppobj, fs, k)
  hx = h.x
  hy = h.y
  hl = hy
  hu = hy
  ha = hy^2
  nsim = cols(obj.xmat)
  ;
  i = 2
  while (i <= nsim)
    pp = obj.xmat[,i] ~ obj.ymat[,i]
    ppobj = list (pp, area, type)
    h = SPPPkfn (ppobj, fs, k)
    hy = h.y
    hl = min (hl ~ hy, 2)
    hu = max (hu ~ hy, 2)
    ha = ha + hy^2
    i = i + 1
  endo
  ;
  aver = sqrt(ha / nsim)
  res = list(hx, hl, hu, aver)
endp
