proc (res) = SPPPkfn (pp, fs, k)
;
; -----------------------------------------------------------------------
; Library      spatial 
; -----------------------------------------------------------------------
; See_also     SPPPinit SPPPinitrandom SPPPsetregion SPPPgetregion
;              SPPPpsim SPPPstrauss SPPPssi SPPPkenvl SPPPkaver
; -----------------------------------------------------------------------
; Macro        SPPPkfn
; -----------------------------------------------------------------------
; Description  Computes K-fn of a point pattern. Actually, it 
;              computes L = sqrt(K / pi). Note that SPPPinit or
;              SPPPsetregion must have been called before.
; -----------------------------------------------------------------------
; Usage        ppkfn = SPPPkfn (pp, fs, k)
; Input
;   Parameter  pp
;   Definition list   -  (of type "PP") calculated by SPPPinit
;   Parameter  fs
;   Definition scalar -  full scale of the plot
;   Parameter  k
;   Definition scalar -  number of regularly spaced distances in (0, fs)
; Output
;   Parameter  res
;   Definition list   -  consisting of components 
;                       x, y, k, dmin, lm --
;   Parameter  x
;   Definition n x 1  -  vector of distances
;   Parameter  y
;   Definition n x 1  -  vector of L-fn values
;   Parameter  k
;   Definition scalar -  number of distances returned (which may be less 
;                       than k if fs is too large)
;   Parameter  dmin
;   Definition scalar -  minimum distance between pair of points
;   Parameter  lm
;   Definition scalar -  maximum deviation from L(t) = t
; -----------------------------------------------------------------------
; Notes        C-Code from Venables, Ripley (1999) forms the basis of 
;              this quantlet. Also check this reference for more details.
; -----------------------------------------------------------------------
; Example      ; load the spatial statistics library
;	       library ("spatial")
;              ; read a spatial data set
;	       pines = read ("pines.dat")
;	       ; initialize a point process
;  	       pinesobj = SPPPinit (pines, 0, 96, 0, 100, 10)
;              ; calculate K-fn in interval (0, 5), split into 100 segments
;              ppkfn = SPPPkfn (pinesobj, 5, 100)
; -----------------------------------------------------------------------
; Result       Returns the L-fn for a point pattern, including some of
;              its major characteristics.
; -----------------------------------------------------------------------
; Reference    Venables, W. N., Ripley, B. D. (1999): Modern Applied
;              Statistics with S-Plus, Third Edition, Springer, New York 
; -----------------------------------------------------------------------
; Link         ../tutorials/spatialstart.html Spatial statistics tutorial
; -----------------------------------------------------------------------
; Author       Juergen Symanzik, 000725
; -----------------------------------------------------------------------
;
  error (exist("pp") != 9, "pp must be a list")
  error (comp(pp, "type") == 0, "pp must contain component type")
  error (pp.type != "PP", "pp.type must be PP")
  error (comp(pp, "pp") == 0, "pp must contain component pp")
  error (cols(pp.pp) < 2, "pp.pp must have at least 2 columns")
  error (exist("fs") != 1, "fs must be numeric")
  error (fs <= 0, "fs must be > 0")
  error (exist("k") != 1, "k must be numeric")
  error (k < 1, "k must be >= 1")
  ;
  ;
  area = SPPPgetregion ()
  pp1 = pp.pp[,1]
  pp2 = pp.pp[,2]
  lppx = rows(pp.pp)
  k1 = k
  h = 1:k
  res = 1:4
  dmin = 1
  lm = 1
  ;
  os = getenv ("os")
  if (os == "windows")
    d = dlcall ("_VR_sp_pp2", pp1, pp2, lppx, k1, h, dmin, lm, fs)
  else  
    ; Unix
    d = dlcall ("VR_sp_pp2", pp1, pp2, lppx, k1, h, dmin, lm, fs)
  endif
  ;
  y = h[1:k1]
  x = (1:k1) * fs / k
  lm = max (dmin | lm)
  res = list (x, y, k, dmin, lm)
endp
