proc(m,b,const) = backfit(t,y,h,loc,kern,opt)
; -----------------------------------------------------------------
;   Library      gam
; -----------------------------------------------------------------
;   See_also     intest, intestpl, gamfit, knn
; -----------------------------------------------------------------
;   Macro        backfit
; -----------------------------------------------------------------
;   Description  the estimates for the components of an additive
;                (partial linear) model are calculated. If the local
;                linear smoother is applied, the first derivatives are
;                calculated as well, additionally the second derivatives
;                if the local quadratic smoother is chosen.
; -----------------------------------------------------------------
;   Usage        {m,b,const} = backfit(t,y,h,loc,kern{,opt})
;   Input
;     Parameter   t
;     Definition      n x p matrix, the observed continuous
;                     explanatory variable
;     Parameter   y
;     Definition      n x 1 matrix, the observed response variable
;     Parameter   h
;     Definition      p x 1 vector or scalar, the bandwidth if loc>-1,
;                     else the second parameter of knn.
;     Parameter   loc
;     Definition      {-1,0,1,2}, if loc>-1, the degree of the chosen
;                     local polynomial smoother else the knn is chosen.
;     Parameter   kern
;     Definition      string, the kernel to be used.
;                     If loc=-1 it has no meaning.
;     Parameter   opt.x
;     Definition      n x d matrix, optional, the observed discrete
;                     explanatory variables (linear part)
;     Parameter   opt.miter
;     Definition      integer, maximal number of iterations. The default
;                     is 50.
;     Parameter   opt.cnv
;     Definition      integer, convergence criterion. The default is 1.0e-6.
;     Parameter   opt.shf
;     Definition      integer,  (show-how-far) if exists and =1, an output
;                     is produced which indicates how the iteration
;                     is going on (additive function / point of estimation /
;                     number of iteration).
;   Output
;     Parameter   m
;     Definition      n x pp matrix, where pp is p*(loc+1). The estimates
;                     of the additive components are given in column 1 to p,
;                     the first derivatives in column (p+1) to (2p) and
;                     the second derivatives in (2p+1) to (3p).
;     Parameter   b
;     Definition      d x 1 vector, parameter estimate of the linear part
;     Parameter   const
;     Definition      scalar, estimate of the constant
; -----------------------------------------------------------------
;   Example   library("gam")
;             randomize(1)
;             n   = 100
;             t   = normal(n,2)             ; explanatory variable
;             x   = normal(n,2)             ; the linear part
;             f1  = - sin(2*t[,1])          ; estimated functions
;             f2  = t[,2]^2
;             eps = normal(n,1) * sqrt(0.75)
;             y   = x[,1] - x[,2]/4 + f1 + f2 +eps      ; response variable
;             h   = 0.5
;             opt = gamopt("x",x,"shf",1)   ; the linear part is used
;             ;                               and the iterations will be shown
;             {m,b,const} = backfit(t,y,h,0,"qua",opt)
;             ;
;             b                             ; coefficients for the linear part ([1, -1/4] were used)
;             const                         ; estimation of the constant
;             ;
;             pic = createdisplay(1,2)      ; preparing the graphical output
;             d1  = t[,1]~m[,1]
;             d2  = t[,2]~m[,2]
;             setmaskp(d1,4,4,4)
;             setmaskp(d2,4,4,4)
;             m1  = mean(f1)
;             m2  = mean(f2)
;             yy  = y - x*b - const
;             x1  = t[,1]~(yy - m[,2])
;             x2  = t[,2]~(yy - m[,1])
;             setmaskp(x1,1,11,4)
;             setmaskp(x2,1,11,4)
;             setmaskl(d1,(sort(d1~(1:rows(d1)))[,3])',4,1,1)
;             setmaskl(d2,(sort(d2~(1:rows(d2)))[,3])',4,1,1)
;             show(pic,1,1,d1,x1,t[,1]~(f1-m1))
;             show(pic,1,2,d2,x2,t[,2]~(f2-m2))
; -----------------------------------------------------------------
;   Result    estimates of the additive functions using backfitting,
;             see 'Generalized Additive Models by Hastie and Tibshirani'
;             (1990)
; -----------------------------------------------------------------
;   Author    Sperlich 980217
; -----------------------------------------------------------------
 cy=cols(y)
 ry=rows(y)
 ch=cols(h)
 rh=rows(h)
 error(cy>1 && ry>1,"backfit: y must be a vector")
 error(ch>1 && rh>1,"backfit: h must be a vector or a scalar")
 if (cy>1)
  y=y'
 endif
 if (ch>1)
  h=h'
  rh=ch
 endif
 p = cols(t)
 n = rows(t)
 error(rows(y)<>n,"backfit: y must have the same number of rows as t")
 error(rh>1 && rh<>p,"backfit: number of rows of h must be the same as number of columns of t")
 error((rint(loc)<>loc)||(loc<-1)||(loc>2),"backfit: loc must be equal to -1, 0, 1 or 2")
 if (comp("opt","shf")>0)
   shf = opt.shf
 else
   shf = 0
 endif
 if (comp("opt","miter")>0)
   maxit = opt.miter
 else
   maxit = 50
 endif
 if (comp("opt","cnv")>0)
   concr = opt.cnv
 else
   concr = 1e-6
 endif
 if (loc==-1)
   m = 0*matrix(n,p)
 else
   m = 0*matrix(n,p*(loc+1))
 endif
 yr = y
 if (comp("opt","x")>0)
   d = cols(opt.x)
   b = matrix(d+1)
   x1 = matrix(n)~opt.x
 else
   d = 0
   b = matrix(1)
   x1 = matrix(n)
 endif
 if (rows(h)==1)
   h = h[1]*matrix(p,1)
 endif
 lq = inv(x1'*x1) * x1'
 it = 0
 do
   it = it + 1
   if (shf)
     ("backfit: "+string("%1.0f",it))
   endif
   mold = m
   bold = b
   yr = y - sum(m[,1:p],2)
   b = lq*yr
   yb = y - x1*b
   j = 1
   switch
   case (loc==0)
    while (j.<=p)
     yr = yb - sum(m[,1:p],2) + m[,j]
     ts = sort(t[,j]~yr~(1:n))
     rm = sker(ts[,1],h[j],kern,ts[,2]~matrix(n))
     mh = rm[,1]./rm[,2]
     mh = sort(mh~ts[,3],2)
     m[,j] = replace(mh[,1],NaN,0)
     m[,j] = m[,j] - mean(m[,j])
     j = j+1
    endo
   break
   case (loc==-1)
    while (j.<=p)
     yr = yb - sum(m[,1:p],2) + m[,j]
     ts = sort(t[,j]~yr~(1:n))
     mh = knn(ts[,2],h[j])
     mh = sort(ts[,3]~mh)[,2:(cols(mh)+1)]
     m[,j] = replace(mh[,1],NaN,0)
     m[,j] = m[,j] - mean(m[,j])
     j = j+1
    endo
   break
   default
    while (j.<=p)
     yr = yb - sum(m[,1:p],2) + m[,j]
     ts = sort(t[,j]~yr~(1:n))
     mh = locpol(ts[,1],ts[,1],ts[,2],h[j],loc-1)
     mh = sort(ts[,3]~mh)[,2:(cols(mh)+1)]
     m[,j] = replace(mh[,1],NaN,0)
     m[,j] = m[,j] - mean(m[,j])
     m[,j+p] = mh[,2]
     if (loc==2)
       m[,j+2*p] = mh[,3]*2
     endif
     j = j+1
    endo
   break
   endsw
   nom = sum((m[,1:p]-mold[,1:p])^2) ~  sum( (b-bold)^2 )
   denom = sum(mold[,1:p].*mold[,1:p]) ~  sum( bold^2 )
   crit = max(nom./denom) .< concr
 until( crit || (it>maxit) )                    ; end of main loop
 const = b[1]
 if (rows(b)>1)
   b = b[2:rows(b)]
 else
   b = 0
 endif
endp
